/*
 * mtk - Maths Toolkit for X11
 *
 * Copyright 1994-1997   andrewr@chiark.greenend.org.uk (Andrew Ross)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

// Class to define menus

#include <qapp.h>
#include <qmsgbox.h>
#include <qfiledlg.h>
#include <qstring.h>
#include <qclipbrd.h>
#include <qaccel.h>
#include <stdio.h>
#include "../dialogs/input.h"
#include "../dialogs/setvar.h"
#include "../dialogs/diff.h"
#include "../dialogs/powervar.h"
#include "../dialogs/numint.h"
#include "../dialogs/numode.h"
#include "../dialogs/grpref.h"
#include "../dialogs/graphtab.h"
#include "mainwin.h"

void
Math_Main::setMenu() 
{
  fileMenu = new QPopupMenu;   // File menu
  CHECK_PTR(fileMenu);
  NewID = fileMenu->insertItem("New", this, SLOT(new_()),ALT + Key_N );
  OpenID = fileMenu->insertItem("Open", this, SLOT(open_()), ALT + Key_O );
  SaveID = fileMenu->insertItem("Save", this, SLOT(save()), ALT + Key_S );
  SaveAsID = fileMenu->insertItem("Save As", this, SLOT(save_as()), ALT + Key_A );
  LogToFileID = fileMenu->insertItem("Log to file", this, SLOT(log_to_file()), ALT + Key_L );
  fileMenu->insertSeparator();
  ExitID = fileMenu->insertItem("Exit", this, SLOT(quit_app()), ALT + Key_X );

  editMenu = new QPopupMenu;    // Edit menu
  CHECK_PTR(editMenu);
  CutID = editMenu->insertItem("Cut", this, SLOT(cut()), CTRL + Key_X );
  PasteID = editMenu->insertItem("Paste", this, SLOT(paste()), CTRL + Key_V );
  CopyID = editMenu->insertItem("Copy", this, SLOT(copy()), CTRL + Key_C );
  CopyOutputID = editMenu->insertItem("Copy Output", this, SLOT(CopyOutput()), CTRL + Key_N );
  ClearID  = editMenu->insertItem("Clear", this, SLOT(clear()) ); 

  equationMenu = new QPopupMenu;    // Equation menu
  CHECK_PTR(equationMenu);
  EnterID = equationMenu->insertItem("Enter", this, SLOT(enter()), CTRL + Key_A  );
  EditID = equationMenu->insertItem("Edit", this, SLOT(edit_()), CTRL + Key_E );
  SelectID = equationMenu->insertItem("Select", this, SLOT(select()), CTRL + Key_S );
  DeleteID = equationMenu->insertItem("Delete", this, SLOT(delete_current()), Key_Delete );
  UndoID = equationMenu->insertItem("Undo delete", this, SLOT(undo()), CTRL + Key_Z );
  DeleteAllID = equationMenu->insertItem("Delete All", this, SLOT(delete_all()), CTRL + Key_Delete );
  TidyUpID = equationMenu->insertItem("Tidy equation", this, SLOT(tidy()));
  SetVarID = equationMenu->insertItem("Set Variables", this, SLOT(set_variables()), CTRL + Key_T );

  evaluatingMenu = new QPopupMenu;    // Evaluating menu
  CHECK_PTR(evaluatingMenu);
  EvaluateID = evaluatingMenu->insertItem("Evaluate", this, SLOT(evaluate()), CTRL + Key_Equal );
  IntegrateID = evaluatingMenu->insertItem("Integrate", this, SLOT(integrate()), CTRL + Key_I );
  FindRootID = evaluatingMenu->insertItem("Find Root", this, SLOT(find_root()), CTRL + Key_R );
  DifferentiateID = evaluatingMenu->insertItem("Differentiate", this, SLOT(differentiate()), CTRL + Key_D );
  PowerSeriesID = evaluatingMenu->insertItem("Power Series", this, SLOT(power_series()), CTRL + Key_P );
  OdeID = evaluatingMenu->insertItem("Ord Diff Eq", this, SLOT(ode()), CTRL + Key_O );

  graphMenu = new QPopupMenu;      // Graph menu
  CHECK_PTR(graphMenu);
  GraphID = graphMenu->insertItem("Plot Graph", this, SLOT(plot_graph()), CTRL + Key_G );
  GraphPrefID  = graphMenu->insertItem("Graph Pref", this, SLOT(graph_pref()), CTRL + Key_F );
  DeleteGraphsID = graphMenu->insertItem("Delete graphs", this, SLOT(delete_graphs()) );

  optionsMenu = new QPopupMenu;    // Options menu
  CHECK_PTR(optionsMenu);
  SigFigID = optionsMenu->insertItem("Sig Fig", this, SLOT(sig_fig()) );
  optionsMenu->insertSeparator();
  SciNotID = optionsMenu->insertItem("Sci Notation", this, SLOT(sci_not()) );
  AngleID  = optionsMenu->insertItem("Radians", this, SLOT(angle()) );
  optionsMenu->setCheckable(TRUE);
  optionsMenu->setItemChecked(SciNotID,SciNot);
  optionsMenu->setItemChecked(AngleID,Radians);

  helpMenu = new QPopupMenu;    // Help menu
  CHECK_PTR(helpMenu);
  ContentsID = helpMenu->insertItem("Contents", this, SLOT(contents()), ALT+Key_H );
  helpMenu->insertSeparator();
  AboutID = helpMenu->insertItem("About", this, SLOT(about()) );

  eqnPopup = new QPopupMenu;  // Popup menu for equations
  CHECK_PTR(eqnPopup);
  PopupEnterID = eqnPopup->insertItem("Enter", this, SLOT(enter()) );
  PopupEditID = eqnPopup->insertItem("Edit", this, SLOT(edit_()) );
  PopupDeleteID = eqnPopup->insertItem("Delete", this, SLOT(delete_current()) );
  PopupTidyUpID = eqnPopup->insertItem("Tidy equation", this, SLOT(tidy()));

  grey();    // Grey unavailable menu items
  equationMenu->setItemEnabled(SelectID,FALSE);
  equationMenu->setItemEnabled(DeleteID,FALSE);
  equationMenu->setItemEnabled(DeleteAllID,FALSE);
  graphMenu->setItemEnabled(DeleteGraphsID,FALSE);
  equationMenu->setItemEnabled(UndoID,FALSE);

  menu = new QMenuBar( this );    // Set up menu bar
  CHECK_PTR( menu );
  menu->insertItem( "File", fileMenu );
  menu->insertItem( "Edit", editMenu );
  menu->insertItem( "Equations", equationMenu );
  menu->insertItem( "Evaluating", evaluatingMenu );
  menu->insertItem( "Graph", graphMenu );
  menu->insertItem( "Options", optionsMenu );
  menu->insertItem( "Help", helpMenu );
}

// File|New function
void 
Math_Main::new_()
{
  delete_all();
  delete_graphs();
  InitVariables();
}

// File|Open function
void 
Math_Main::open_()
{
  QString *tempname;
  
  tempname = new QString(QFileDialog::getOpenFileName(0,"*.mtk"));
  if ( tempname->isNull() ) {
    delete tempname;
    return;
  };
  switch ( OpenFile(tempname) ) {
  case -4:
    QMessageBox::message("Error!","This version of mtk file\nis not supported");
  case -3:
    QMessageBox::message("Error!","File is not a mtk file");
    break;
  case -2:
    QMessageBox::message("Error!","Error in reading file");
    break;
  case -1:
    QMessageBox::message("Error!","Unable to open file");
    break;
  case 0:
    if ( LogStream ) {
      *LogStream << "Opened file";
      *LogStream << tempname->data();
      *LogStream << "\n";
    }
    break;
  default:
    QMessageBox::message("Error!","Error in opening file");
    break;
  }
}

// File|Save function
void 
Math_Main::save()
{
  QString *tempname;

  if ( Filename == NULL ) {
    tempname = new QString(QFileDialog::getSaveFileName(0,"*.mtk"));
    if ( tempname->isNull() ) {
      delete tempname;
      return;
    }
    SaveFile(tempname);
  }
  else {
    SaveFile(Filename);
  }  
}

// File|Save_As function
void 
Math_Main::save_as()
{
  QString *tempname;

  tempname = new QString(QFileDialog::getSaveFileName(0,"*.mtk"));
  if ( tempname->isNull() ) {
    delete tempname;
    return;
  }
  SaveFile(tempname);

}

// File|Log To File
void
Math_Main::log_to_file()
{
  QString *tempname;

  if (LogFile) {
    if ( QMessageBox::query( "Log to file", "Are you sure you wish to\nstop logging to file?" ) ) {
      LogFile->close();
      delete LogFile;
      LogFile = NULL;
      delete LogStream;
      LogStream = NULL;
      fileMenu->changeItem("Log to file",LogToFileID);
    };
  }
  else {
    tempname = new QString(QFileDialog::getSaveFileName(0,"*.log"));
    if ( tempname->isNull() ) {
      delete tempname;
      return;
    }
    LogFile = new QFile(tempname->data());
    if (!LogFile->open(IO_WriteOnly)) {
      QMessageBox::message("Error!","Unable to open file");
      delete tempname;
      delete LogFile;
      LogFile = NULL;
      return;
    }
    LogStream = new QTextStream(LogFile);
    fileMenu->changeItem("Stop logging",LogToFileID);
  }
}

// File|Quit function
void
Math_Main::quit_app()
{
  if (LogFile) {
    LogFile->close();
  }
  qApp->quit();
}

// Edit|Cut function
void 
Math_Main::cut()
{
  int i;

  i = CurrentEqu;
  if (i != 0) {
    clip->setText( Equations.at(i - 1)->EquationText );
    Equations.remove(i - 1);
    SelectEqu(0);
    Variable=NULL;
    Variable1=NULL;
    if (Equ2 == i) {
      Equ2=0;
      Variable2=NULL;
    };
    Update();
  }
  else {
    QMessageBox::message("Error!","No equation selected");
  }
}

// Edit|Paste function
void 
Math_Main::paste()
{
  int i,flag;
  char *EqnText;
  Equation *eqn;

  i = 0;
  flag = 0;
  EqnText = (char *)clip->text();
  if (EqnText) {
    try {
      eqn = parser.TextToTree( EqnText );
      eqn->Radians = Radians;
      Equations.append(eqn);
      
      i--;
      eqn->Radians = true;
      SelectEqu(Equations.at()+1);
    }
    catch (error_struct err) {
      ErrorCheck(err);
      return;
    }
  }
  else {
    QMessageBox::message("Error!","No text in clipboard");
  }
}

// Edit|Copy function
void 
Math_Main::copy()
{
  if (CurrentEqu != 0) {
    clip->setText( Equations.at(CurrentEqu - 1)->EquationText );
  }
  else {
    QMessageBox::message("Error!","No equation selected");
  }
}

// Edit|Clear function
void 
Math_Main::clear()
{
  clip->clear();
}

// Equations|Enter function
void 
Math_Main::enter()
{
  bool cancel = false, eqn_ok = false;
  int at;
  Equation *newEqn = NULL;
  
  QString *EqnText = new QString();
  
  Input GetEqn("Enter equation", EqnText,"");
  
  do {
    GetEqn.setValue(EqnText);
    if (GetEqn.exec()==QDialog::Accepted) {
      try {
	newEqn = parser.TextToTree( EqnText->data() );
	eqn_ok = true;
      }
      catch (error_struct err) {
	ErrorCheck(err);
      }
    } 
    else cancel = true;
  }
  while ((!cancel) && (!eqn_ok));
  if (cancel == false) {
    newEqn->Radians = Radians;
    Equations.append( newEqn );
    at = Equations.at() + 1;
    SelectEqu( at );
    if ( LogStream ) {
      *LogStream << "Enter equation ";
      *LogStream << at;
      *LogStream << " = ";
      *LogStream << Equations.current()->EquationText;
      *LogStream << "\n";
    }
    if ( Equations.count() == 1 ) {
      equationMenu->setItemEnabled(SelectID,TRUE);
      equationMenu->setItemEnabled(DeleteID,TRUE);
      equationMenu->setItemEnabled(DeleteAllID,TRUE);
    }
  }
  delete EqnText;
}

// Equations|Edit function
void 
Math_Main::edit_()
{
  int equno;
  bool cancel = false, eqn_ok = false;
  QString *eqntext;
  Equation *tempEqn;
  Equation *eqn;

  equno = CurrentEqu;
  if (equno == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }
  equno--;
  eqn = Equations.at(equno);
  if ( eqn == NULL ) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }

  eqntext = new QString( eqn->EquationText );
  Input EditEqn("Edit equation", eqntext,"");
  do {
    EditEqn.setValue(eqntext);
    if (EditEqn.exec()==QDialog::Accepted) {      
      try {
	tempEqn = parser.TextToTree( eqntext->data() );
	eqn_ok = true;
      }
      catch (error_struct err) {
	ErrorCheck(err);
      }
    }
    else { 
      cancel = true;
    }
  }
  while ( (!cancel) && ( !eqn_ok ) );
  if ( !cancel ) {
    tempEqn->Radians=true;
    Equations.remove(eqn);
    Equations.insert(equno,tempEqn);
    if (LogStream) {
      *LogStream << "Alter equation ";
      *LogStream << equno+1;
      *LogStream << " = ";
      *LogStream << tempEqn->EquationText;
	*LogStream << "\n";
    }
    Update();
  }
  return;
}


// Equations|Select function
void 
Math_Main::select()
{
  QString text;// = new QString();
  int i;
  bool error_flag;

  if ( CurrentEqu > 0 ) {
    text.setNum(CurrentEqu);
  }
  else {
    text = "";
  };

  Input GetEqnNum("Enter equation number to select",&text,"");

  if ( GetEqnNum.exec()==QDialog::Accepted ) {   
    i = text.toInt(&error_flag);
    if ( ( error_flag ) && ( i >= 1) && ( i <= (int) Equations.count() ) ) {
      SelectEqu(i);
      return;
    };
    QMessageBox::message("Error!","Not a valid equation number");
  };
}

// Equation|Delete function
void 
Math_Main::delete_()
{
  QString *text = new QString();
  int i;
  bool error_flag;
  Input GetEqnNum("Enter equation number to delete",text,"");

  if ( GetEqnNum.exec() == QDialog::Accepted ) {   
    i = text->toInt(&error_flag);
     if ( error_flag ) {
       if (oldEqn) delete oldEqn;
       oldEqn = Equations.take(i-1);
       if ( oldEqn ) {
	 if ( LogStream ) {
	   *LogStream << "Delete equation ";
	   *LogStream << i;
	   *LogStream << "\n";
	 }
	 if ( CurrentEqu == i ) {
	   SelectEqu(0);
	   Variable = NULL;
	   Variable1 = NULL;
	 }
	 else {
	   if ( CurrentEqu > i ) {
	     SelectEqu(CurrentEqu-1);
	   }
	 }
	 if ( Equ2 == i ) {
	   Equ2 = 0;
	   Variable2 = NULL;
	 };
	 equationMenu->setItemEnabled(UndoID, TRUE);
	 if ( Equations.count() == 0 ) {
	   equationMenu->setItemEnabled(SelectID,FALSE);
	   equationMenu->setItemEnabled(DeleteID,FALSE);
	   equationMenu->setItemEnabled(DeleteAllID,FALSE);
	 }
	 Update();
	 return;
       }
     }
     QMessageBox::message("Error!","Unable to delete equation");
  }
}

// Popup menu Delete current equation function
void 
Math_Main::delete_current()
{
  int i;

  i = CurrentEqu;
  if (oldEqn) delete oldEqn;
  oldEqn = Equations.take(i-1);
  if ( oldEqn ) {
    if (LogStream) {
      *LogStream << "Delete equation ";
      *LogStream << i;
      *LogStream << "\n";
    }
    if (CurrentEqu > (int) Equations.count())
      SelectEqu(CurrentEqu-1);
    else 
      SelectEqu(CurrentEqu);
    Variable = NULL;
    Variable1 = NULL;
    if ( Equ2 == i ) {
      Equ2 = 0;
      Variable2 = NULL;
    };
    equationMenu->setItemEnabled(UndoID, TRUE);
    if ( Equations.count() == 0 ) {
      equationMenu->setItemEnabled(SelectID,FALSE);
      equationMenu->setItemEnabled(DeleteID,FALSE);
      equationMenu->setItemEnabled(DeleteAllID,FALSE);
    }
    Update();
    return;
  };
  QMessageBox::message("Error!","No equation selected to delete");
}

// Equation|Undo delete function
void
Math_Main::undo()
{
  if (oldEqn) {
    Equations.append(oldEqn);
    SelectEqu( Equations.at() + 1 );
    oldEqn = NULL;
    equationMenu->setItemEnabled(UndoID,FALSE);
    Update();
  }
}

// Equation|Delete All function
void 
Math_Main::delete_all()
{
  int i;

  if ( Equations.isEmpty() ) return;
  if ( QMessageBox::query( "Delete Equations", "Are you sure you want to\ndelete all equations" ) ) {
    i=0;
    SelectEqu(0);
    Variable=NULL;
    Variable1=NULL;
    Variable2=NULL;
    Equations.clear();
    if (LogStream) *LogStream << "Delete all equations\n";
    Update();
    equationMenu->setItemEnabled(SelectID,FALSE);
    equationMenu->setItemEnabled(DeleteID,FALSE);
    equationMenu->setItemEnabled(DeleteAllID,FALSE);
  }; 
}

// Equation|Tidy function
void 
Math_Main::tidy()
{
  Equation *eqn;

  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }

  eqn = Equations.at(CurrentEqu-1);
  parser.TidyUpEqn(eqn);
  parser.TreeToText(eqn);
  Update();
}

// Equation|Set Variable function
void 
Math_Main::set_variables()
{
  Equation *eqn;

  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }

  eqn = Equations.at(CurrentEqu-1);
  if ( eqn->Variables.count() != 0 ) {
    SetVar SetVarVal( eqn, LogStream, &parser );
    if (LogStream) {
      *LogStream << "Set variables for equation ";
      *LogStream << CurrentEqu;
      *LogStream << "\n";
    }
    SetVarVal.exec();
  }
  else {
    QMessageBox::message("Error!","No variables in this equation");
  };
}

// Evaluating|Evaluate function
void 
Math_Main::evaluate()
{
  double answer;
  Equation * eqn;

  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }
  
  eqn = Equations.at(CurrentEqu-1);
  if ( eqn != NULL) {
    if (eqn->Variables.count()) {
      SetVar SetVarVal( eqn, LogStream, &parser );
      SetVarVal.exec();
    }
    try {
      answer = eqn->Evaluate();
      if (LogStream) {
	*LogStream << "Evaluate equation ";
	*LogStream << CurrentEqu;
      }
      NumberToOutput(answer);
    }
    catch (error_struct err) {
      ErrorCheck(err);
    }
  }
}

// Evaluating|Integrate function
void 
Math_Main::integrate()
{
  int var;
  Equation *eqn;
  double answer;
  
  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }
  
  eqn = Equations.at( CurrentEqu - 1);

  if (eqn->Variables.count() == 0) {
    QMessageBox::message("Error!","No variable to integrate with");
    return;
  }

  NumInt DoNumInt( &var, eqn, &(Range), "Integrate numerically", "Range", "Step size", &parser );
  if ( DoNumInt.exec() == QDialog::Accepted ) {
    try {
      answer = eqn->NumInt( Range , var );
    }
    catch (error_struct err) {
      ErrorCheck(err);
      return;
    }
    if (LogStream) {
      *LogStream << "Integrate equation ";
      *LogStream << CurrentEqu;
    }
    NumberToOutput( answer );
  }
}

// Evaluating|Find Root function
void 
Math_Main::find_root()
{
  int var;
  Equation *eqn;
  double answer,step;

  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }
  
  eqn = Equations.at( CurrentEqu - 1 );

  if (eqn->Variables.count() == 0) {
    QMessageBox::message("Error!","No variable to find root with");
    return;
  }

  step = Range.Step;
  Range.Step = Range.Accuracy;
  NumInt DoFindRoot( &var, eqn, &(Range), "Find root", "Interval", "Accuracy", &parser );
  if ( DoFindRoot.exec() == QDialog::Accepted ) {
    Range.Accuracy = Range.Step;
    Range.Step = step;
    try {
      answer = eqn->FindRoot( Range , var );
    }
    catch (error_struct err) {
      ErrorCheck(err);
      Range.Accuracy = Range.Step;
      return;
    }
    if (LogStream) {
      *LogStream << "Root of equation ";
      *LogStream << CurrentEqu;
      *LogStream << " in the range ";
      *LogStream << Range.From;
      *LogStream << " < ";
      *LogStream << eqn->VarText(var);
      *LogStream << " < ";
      *LogStream << Range.To;
    }
    NumberToOutput( answer );
  }
  else Range.Accuracy = Range.Step;
}

// Evaluating|differentiate function
void 
Math_Main::differentiate()
{
  int var,at;
  Equation *eqn, *CurrEqu;

  if ( CurrentEqu==0 ) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }

  CurrEqu = Equations.at(CurrentEqu-1);

  if (CurrEqu->Variables.count() == 0) {
    QMessageBox::message("Error!","No variable to differentiate with");
    return;
  }

  Diff GetDiffVar( &var , CurrEqu );
  if ( GetDiffVar.exec() == QDialog::Accepted ) {
    try {
      eqn = new Equation();
      CurrEqu->Differentiate(eqn,var,&parser);
    }
    catch (error_struct err) {
      ErrorCheck(err);
      return;
    }
    Equations.append(eqn);
    at = Equations.at()+1;
    if (LogStream) {
      *LogStream << "Differentiate equation ";
      *LogStream << CurrentEqu;
      *LogStream << " with respect to ";
      *LogStream << CurrEqu->VarText(var);
      *LogStream << "\n";
      *LogStream << "Equation ";
      *LogStream << at;
      *LogStream << " = ";
      *LogStream << eqn->EquationText;
      *LogStream << "\n";
    }
    SelectEqu(at);
    Update();
  };
}

// Evaluating|Power Series function
void 
Math_Main::power_series()
{
  int var,at;
  uint terms;
  Equation *CurrEqu = Equations.at(CurrentEqu-1);
  Equation *eqn;

  if ( CurrentEqu==0 ) {
    QMessageBox::message("Error!","No equation selected");
    return;
  };

  if ( CurrEqu->Variables.count() == 0 ) {
    QMessageBox::message("Error!","No variables to create power series in");
    return;
  };
    
  PowerVar GetPowerVar( CurrEqu, &var, &terms, &parser);
  if ( GetPowerVar.exec() == QDialog::Accepted ) {
    try {
      eqn = new Equation();
      CurrEqu->PowerSeries(eqn, var, terms, &parser);
      Equations.append(eqn);
      at = Equations.at()+1;
      if (LogStream) {
	*LogStream << "Power series of equation ";
	*LogStream << CurrentEqu;
	*LogStream << " about ";
	*LogStream << CurrEqu->VarText(var);
	*LogStream << " = ";
	*LogStream << CurrEqu->VarValue(var);
	*LogStream << "\n";
	*LogStream << "Equation ";
	*LogStream << at;
	*LogStream << " = ";
	*LogStream << eqn->EquationText;
	*LogStream << "\n";
      }
      SelectEqu(at);
      Update();
    }
    catch (error_struct err) {
      ErrorCheck(err);
    }
  }
}

// Evaluating|ODE function
void 
Math_Main::ode()
{
  int var_x,var_y;
  Equation *eqn;
  double answer,acc;
  
  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }  
  
  eqn = Equations.at( CurrentEqu - 1 );
  acc = Range.Accuracy;
  NumODE NumODE( &var_x, &var_y, &(Range), eqn, &parser);
  if ( NumODE.exec() == QDialog::Accepted ) {
    try {
      answer = eqn->RK4( Range, var_x, var_y );
    }
    catch (error_struct err) {
      ErrorCheck(err);
      Range.Accuracy = acc;
      return;
    }
    if (LogStream) {
      *LogStream << "Numerical solution of dy/dx = equation ";
      *LogStream << CurrentEqu;
      *LogStream << " ('y' = ";
      if (var_y) {
	*LogStream << eqn->VarText(var_y);
      }
      else {
	*LogStream << " NULL";
      }
      *LogStream << ", 'x' = ";
      if (var_x) {
	*LogStream << eqn->VarText(var_x);
      }
      else {
	*LogStream << " NULL";
      }
      *LogStream << ")\n from ";
      *LogStream << Range.From;
      *LogStream << " to ";
      *LogStream << Range.To;
    }
    NumberToOutput( answer );
  };
  Range.Accuracy = acc;
}

// Graph|Plot graph - get options and call parent function to plot graph
void 
Math_Main::plot_graph()
{
  int equ2, var;

  if (CurrentEqu == 0) {
    QMessageBox::message("Error!","No equation selected");
    return;
  }

  equ2 = ( (Equ2) ? (Equ2) : (CurrentEqu) );

  GraphTabDlg DoGraph( &(GraphType), &Equations, &var, CurrentEqu, &equ2, &parser, LogStream );
  if ( DoGraph.exec()==QDialog::Accepted ) {
    Equ2 = equ2;
    DrawGraph(var);
    graphMenu->setItemEnabled(DeleteGraphsID,TRUE);
  }
  
}

// Graph|Graph Pref function
void 
Math_Main::graph_pref()
{
  GraphPrefDlg GrPrefDlg( &GraphType, &parser );
  GrPrefDlg.exec();
}

// Graph|Delete Graphs function
void
Math_Main::delete_graphs()
{
  GraphWin *gr;

  gr = Graph.first();
  while ( gr  ) {
    gr->close();
    gr = Graph.next();
  }
  Graph.clear();
  graphMenu->setItemEnabled(DeleteGraphsID,FALSE);

}

// Delete graphs and tidy up
void 
Math_Main::delete_graph(bool closeWin, GraphWin *gr)
{
  if (closeWin)
    gr->close();
  Graph.remove(gr);

  if (Graph.count() == 0)
    graphMenu->setItemEnabled(DeleteGraphsID,FALSE);
}

// Options|Sig Fig function
void 
Math_Main::sig_fig()
{
  int i;
  bool error_flag;
  QString text;
  text.setNum(SigFig);
  Input GetSigFig("Enter the number of significant figures",&text,"");

  if ( GetSigFig.exec() == QDialog::Accepted ) {
    i = text.toInt(&error_flag);
    if ( (i>0) && (i<=15) ) {
      SigFig = i;
      parser.setSigFig(i);
      if (LogStream) {
	*LogStream << i;
	*LogStream << " significant figures\n";
      }
    }
    else {
      QMessageBox::message("Error!","Number of significant figures must be between 1 and 15");
    };
  };
}

// Options|Sci Notation function
void 
Math_Main::sci_not()
{
  SciNot = !SciNot;
  parser.setSciNot(SciNot);
  optionsMenu->setItemChecked(SciNotID,SciNot);
  if ( LogStream ) {
    if ( SciNot ) {
      *LogStream << "Scientific notation on\n";
    }
    else {
      *LogStream << "Scientific notation off\n";
    }
  }
}

// Options|Angular units function
void 
Math_Main::angle()
{
  int i;

  Radians = !Radians;
  optionsMenu->setItemChecked(AngleID,Radians);    
  if ( LogStream ) {
    if ( Radians ) {
      *LogStream << "Radians\n";
    }
    else {
      *LogStream << "Degrees\n";
    }
  }
  i = 0;
  Equations.first();
  while ( Equations.current() ) {
    Equations.current()->Radians = Radians;
    Equations.next();
  }
}

// Help|Contents function
void 
Math_Main::contents()
{
  QMessageBox::message("Help","Sorry, no online help yet");
}

// Help|About function
void 
Math_Main::about()
{
  QMessageBox::message( "About", "Maths Toolkit\n\nVersion 1.3.1b1\n\n 1994-1997 Andrew Ross.\n\nThis software is covered by the GPL.\nSee the COPYING file distributed\nwith it for more details." );
}

// Grey out options not selectable
void 
Math_Main::grey()
{
  editMenu->setItemEnabled(CutID,FALSE);
  editMenu->setItemEnabled(CopyID,FALSE);
  equationMenu->setItemEnabled(SetVarID,FALSE);
  equationMenu->setItemEnabled(EditID,FALSE);
  equationMenu->setItemEnabled(TidyUpID,FALSE);
  evaluatingMenu->setItemEnabled(EvaluateID,FALSE);
  evaluatingMenu->setItemEnabled(IntegrateID,FALSE);
  evaluatingMenu->setItemEnabled(FindRootID,FALSE);
  evaluatingMenu->setItemEnabled(DifferentiateID,FALSE);
  evaluatingMenu->setItemEnabled(PowerSeriesID,FALSE);
  evaluatingMenu->setItemEnabled(OdeID,FALSE);
  graphMenu->setItemEnabled(GraphID,FALSE);
  eqnPopup->setItemEnabled(PopupEditID,FALSE);
  eqnPopup->setItemEnabled(PopupDeleteID,FALSE);
  eqnPopup->setItemEnabled(PopupTidyUpID,FALSE);
}

// Ungrey items when they can be selected
void 
Math_Main::ungrey()
{
  editMenu->setItemEnabled(CutID,TRUE);
  editMenu->setItemEnabled(CopyID,TRUE);
  equationMenu->setItemEnabled(SetVarID,TRUE);
  equationMenu->setItemEnabled(EditID,TRUE);
  equationMenu->setItemEnabled(TidyUpID,TRUE);
  evaluatingMenu->setItemEnabled(EvaluateID,TRUE);
  evaluatingMenu->setItemEnabled(IntegrateID,TRUE);
  evaluatingMenu->setItemEnabled(FindRootID,TRUE);
  evaluatingMenu->setItemEnabled(DifferentiateID,TRUE);
  evaluatingMenu->setItemEnabled(PowerSeriesID,TRUE);
  evaluatingMenu->setItemEnabled(OdeID,TRUE);
  graphMenu->setItemEnabled(GraphID,TRUE);
  eqnPopup->setItemEnabled(PopupEditID,TRUE);
  eqnPopup->setItemEnabled(PopupDeleteID,TRUE);
  eqnPopup->setItemEnabled(PopupTidyUpID,TRUE);

}


