
/*
 * mtk - Maths Toolkit for X11
 *
 * Copyright 1994-1997   andrewr@chiark.greenend.org.uk (Andrew Ross)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

/* Main window class*/

#define ERROR_MSGS 

#include "mainwin.h"
#include "eqnframe.h"
#include "../equations/errors.h"
#include "../equations/func.h"
#include "graphwin.h"
#include "iconbar.h"

#include <qmsgbox.h>
#include <qdir.h>
#include <qfile.h>
#include <qdstream.h>
#include <qtstream.h>
#include <qclipbrd.h>
#include <qapp.h>
#include <qtooltip.h>

// Needed for OS/2 port
#ifdef __EMX__
#include <math.h>
#include <float.h>
#else
#include <nan.h>
#endif

#undef ERROR_MSGS 

const uint fileVersion = 2;

QDataStream &
operator<<( QDataStream &stream, const bool &b)
{
  return stream << (char) (b?1:0);
}

QDataStream &
operator>>( QDataStream &stream, bool &b)
{
  char c;
  stream >> c;
  b = c?true:false;
  return stream;
}

const char *config_details[26] = {
  "radians","scinot","sigfig","from","to","step","acc","varmin","varmax",
  "xmin","xmax","ymin","ymax","cartesian","parametric","replot",
  "auto","xtic","ytic","xgrid","ygrid","printer","printcmd","paper",
  "orientation","printfile"};

Math_Main::Math_Main(QWidget *parent, char *config, char *file, const char *name) : QFrame(parent, name)
{
// Set up variables for equations
  int w,h;
  QLabel *label;

  // Set up printer
  printer = new QPrinter;


  // Initalise variables or read from file
  //  flag = 0;
  LogFile = NULL;
  LogStream = NULL;
  Graph.setAutoDelete(TRUE);
  oldEqn = NULL;
  LoadDefaults( config );
  Equations.setAutoDelete(TRUE);
  InitVariables();

  // Set up main window
  setMenu();    // Set up menu
  clip = QApplication::clipboard();       // Get clipboard pointer
  setMinimumSize( 500, 400 );   // Set window min size
  resize(size());
  setFrameStyle( QFrame::Box | QFrame::Plain );
  
  w = width();
  h = height();
  Icons = new IconBar(this);     // Set up icon bar
  Icons->move(30,35);
  connect( Icons, SIGNAL(clicked(int)), SLOT(icon_pressed(int)) );
  w -= 60;
  label = new QLabel("", this);       // Set up labels
  label->setText("Equations");
  label->setGeometry(30,80,w,20);
  outlabel = new QLabel("",this);
  outlabel->setText("Output");
  outlabel->setGeometry(30,h-70,w ,20);
  output = new QLabel("" , this);
  output->setGeometry( 30, h-50, w, 30 );
  output->setAlignment( AlignVCenter | AlignLeft );
  output->setFrameStyle( QFrame::Panel | QFrame::Raised );
  output->setBackgroundColor( QColor(210,182,182) );
  QToolTip::add(output,"Output from calculations");
  EqnFrame = new EquationFrame(this, &Equations);   // Set up equation frame
  EqnFrame->setGeometry(30, 105, w, h-145);
  connect( EqnFrame, SIGNAL(buttonpressed(QPoint, int,int )) , SLOT(mouse(QPoint, int, int)) );
  
  // No file specified 
  if ( file != NULL ) 
    Filename = new QString(file);
  else 
    Filename = NULL;
  if ( Filename ) {
    if ( OpenFile(Filename) != 0 ) {
      QMessageBox::message("Error!","Unable to open file");
      InitVariables();
    }
  }
}

// Destructor for class
Math_Main::~Math_Main()
{
  delete_graphs();
  delete fileMenu;
  delete equationMenu;
  delete evaluatingMenu;
  delete graphMenu;
  delete optionsMenu;
  delete helpMenu;
  delete menu;
  delete output;
  delete EqnFrame;
}

// Load initial values from config file or set up defaults
void Math_Main::LoadDefaults( char *config )
{
  QFile configFile;
  QString Line, WorkLine;
  bool noerror,error = false;
  short int tmpshort;
  int tmpint,i;
  double tmpdouble;

  Defaults.Radians = true;
  Defaults.SciNot = false;
  Defaults.SigFig = 8;
  Defaults.Range.From = -1;
  Defaults.Range.To = 1;
  Defaults.Range.Step = 0.1;
  Defaults.Range.Accuracy = 0.001;
  Defaults.GraphType.varmin = -1;
  Defaults.GraphType.varmax = 1;
  Defaults.GraphType.xmin = -1;
  Defaults.GraphType.xmax = 1;
  Defaults.GraphType.ymin = -1;
  Defaults.GraphType.ymax = 1;
  Defaults.GraphType.Cartesian = true;
  Defaults.GraphType.Parametric = false;
  Defaults.GraphType.Replot = false;
  Defaults.GraphType.Auto=true;
  Defaults.GraphType.x_tic_space = 0;
  Defaults.GraphType.y_tic_space = 0;
  Defaults.GraphType.x_grid_space = -1;
  Defaults.GraphType.y_grid_space = -1;

  // No file specified 
    if (config==NULL) {       
      QDir homeDir = QDir::home();
      configFile.setName( homeDir.filePath(".mtkrc") );
  }
  // File specified on command line
  else {       
    configFile.setName( config ); 
    if ( !configFile.exists() ) {
      QMessageBox::message("Error!","Configuration file does not exist");
      return;
    }
  }
  
  if ( configFile.open(IO_ReadOnly) ) {
    QTextStream configStream( &configFile );
    while ( ! configStream.eof() ) {
      Line = configStream.readLine();
      Line.simplifyWhiteSpace();
      WorkLine = Line.lower();
      if ( WorkLine[0] != '#' && !WorkLine.isEmpty() ) {
	i = 0;
	while ( ( i < 26 ) && ( WorkLine.find( config_details[i] ) != 0 ) ) {
	  i++;
	}
	if ( i < 26 ) {
	  WorkLine.remove( 0 , strlen( config_details[i] ) + 1 );
	  if ( !WorkLine.isEmpty() ) {
	    noerror = true;
	    switch ( i ) {
	      // Radians
	    case 0 : 
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) Defaults.Radians = tmpshort;
	      else error = true;
	      break;
	      // Scientific notation
	    case 1 :
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) Defaults.SciNot = tmpshort;
	      else error = true;
	      break;
	      // Significant figures
	    case 2 :
	      tmpint = WorkLine.toInt(&noerror);
	      if ( (tmpint <= 0) || (tmpint > 15) ) noerror = false;
	      if ( noerror ) Defaults.SigFig = tmpint;
	      else error = true;
	      break;
	      // Range.From
	    case 3 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.Range.From = tmpdouble;
	      else error = true;
	      break;
	      // Range.To
	    case 4 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.Range.To = tmpdouble;
	      else error = true;
	      break;
	      // Step size
	    case 5 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( tmpdouble <= 0) noerror = false;
	      if ( noerror ) Defaults.Range.Step = tmpdouble;
	      else error = true;
	      break;
	      // Accuracy
	    case 6 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( tmpdouble <= 0) noerror = false;
	      if ( noerror ) Defaults.Range.Accuracy = tmpdouble;
	      else error = true;
	      break;
	      // varmin
	    case 7 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.varmin = tmpdouble;
	      else error = true;
	      break;
	      // varmax
	    case 8 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.varmax = tmpdouble;
	      else error = true;
	      break;
	      // xmin
	    case 9 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.xmin = tmpdouble;
	      else error = true;
	      break;
	      // xmax
	    case 10 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.xmax = tmpdouble;
	      else error = true;
	      break;
	      // ymin
	    case 11 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.ymin = tmpdouble;
	      else error = true;
	      break;
	      // ymax
	    case 12 :
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.ymax = tmpdouble;
	      else error = true;
	      break;
	      // Cartesian graph
	    case 13 :
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) Defaults.GraphType.Cartesian = tmpshort;
	      else error = true;
	      break;
	      // Parametric graph
	    case 14 :
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) GraphType.Parametric = tmpshort;
	      else error = true;
	      break;
	      // Replot graph
	    case 15 :
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) Defaults.GraphType.Replot = tmpshort;
	      else error = true;
	      break;
	      // Auto scale graph
	    case 16 :
	      tmpshort = WorkLine.toShort(&noerror);
	      if ( (tmpshort != 0) && (tmpshort != 1) ) noerror = false;
	      if ( noerror ) Defaults.GraphType.Auto = tmpshort;
	      else error = true;
	      break;
	      // x tic space
	    case 17:
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror && (tmpdouble >= 0) ) Defaults.GraphType.x_tic_space = tmpdouble;
	      else error = true;
	      break;
	      // y tic space
	    case 18:
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror && (tmpdouble >= 0) ) Defaults.GraphType.y_tic_space = tmpdouble;
	      else error = true;
	      break;
	      // x grid space
	    case 19:
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.x_grid_space = tmpdouble;
	      else error = true;
	      break;
	      // y grid space
	    case 20:
	      tmpdouble = WorkLine.toDouble(&noerror);
	      if ( noerror ) Defaults.GraphType.y_grid_space = tmpdouble;
	      else error = true;
	      break;
	      // Printer
	    case 21 :
	      printer->setPrinterName(WorkLine);
	      break;
	      // Printer command
	    case 22 :
	      printer->setPrintProgram(WorkLine);
	      break;
	      // Paper size
	    case 23 :
	      if ( stricmp( WorkLine , "a4" ) == 0 ) {
		printer->setPageSize( QPrinter::A4 );
		break;
	      }
	      if ( stricmp( WorkLine , "b5" ) == 0 ) {
		printer->setPageSize( QPrinter::B5 );
		break;
	      }
	      if ( stricmp( WorkLine , "letter" ) == 0 ) {
		printer->setPageSize( QPrinter::Letter );
		break;
	      }
	      if ( stricmp( WorkLine , "legal" ) == 0 ) {
		printer->setPageSize( QPrinter::Legal );
		break;
	      }
	      if ( stricmp( WorkLine , "executive" ) == 0 ) {
		printer->setPageSize( QPrinter::Executive );
		break;
	      }
	      error = true;
	      noerror = false;
	      break;
	      // Paper orientation
	    case 24 :
	      if ( stricmp( WorkLine , "portrait" ) == 0 ) {
		printer->setOrientation( QPrinter::Portrait );
		break;
	      }
	      if ( stricmp( WorkLine , "landscape" ) == 0 ) {
		printer->setOrientation( QPrinter::Landscape );
		break;
	      }
	      error = true;
	      noerror = false;
	      break;
	      // Print to file
	    case 25 :
	      printer->setOutputFileName( WorkLine );
	      break;
	    }
	    if (!noerror) {
	      printf("Error parsing value in line : %s\n", Line.data());
	    }
	  }
	  else {
	    error = true;
	    printf("No value in line : %s\n", Line.data());
	  }
	}
	else {
	  error = true;
	  printf("Unknown option in line : %s\n", Line.data());
	}
      }
    }
    if ( Defaults.Range.From > Defaults.Range.To ) {
      Defaults.Range.From = -1;
      Defaults.Range.To = 1;
      error = true;
      printf("from value greater than to value\n");
    }
    if ( Defaults.GraphType.xmin > Defaults.GraphType.xmax ) {
      Defaults.GraphType.xmin = -1;
      Defaults.GraphType.xmax = 1;
      error = true;
      printf("xmin value greater than xmax to value\n");
    }
    if ( Defaults.GraphType.ymin > Defaults.GraphType.ymax ) {
      Defaults.GraphType.ymin = -1;
      Defaults.GraphType.ymax = 1;
      printf("ymin value greater than ymax to value\n");
      error = true;
    }
    if ( error ) QMessageBox::message("Error!","There are errors in the configuration file");
    configFile.close();
  }
}

  

// Initialise all the various variables
void 
Math_Main::InitVariables()
{
  Radians = Defaults.Radians;
  SciNot = Defaults.SciNot;
  SigFig = Defaults.SigFig;
  parser.setSigFig(SigFig);
  parser.setSciNot(SciNot);
  Range = Defaults.Range;
  GraphType = Defaults.GraphType;
  Auto = Defaults.Auto;
  CurrentEqu = 0;
  Equ2 = 0;
  Graph.clear();
  Filename = NULL;
}

// Mouse event in equation label
void 
Math_Main::mouse( QPoint point, int button, int id )
{
  switch ( button ) {
  case LeftButton :
    SelectEqu(id);
    break;
  case RightButton :
    SelectEqu(id);
    eqnPopup->popup( point );
  };
}

// Icon button pressed
void
Math_Main::icon_pressed(int button)
{
  if ( ( Equations.count() == 0 ) && ( button != ENTER_BUTTON ) ) return;
  switch (button) {
  case ENTER_BUTTON:
    enter();
    break;
  case EDIT_BUTTON:
    edit_();
    break;
  case DELETE_BUTTON:
    delete_current();
    break;
  case TIDY_BUTTON:
    tidy();
    break;
  case EVALUATE_BUTTON:
    evaluate();
    break;
  case INTEGRATE_BUTTON:
    integrate();
    break;
  case ROOT_BUTTON:
    find_root();
    break;
  case DIFF_BUTTON:
    differentiate();
    break;
  case POWER_BUTTON:
    power_series();
    break;
  case ODE_BUTTON:
    ode();
    break;
  case GRAPH_BUTTON:
    plot_graph();
    break;
  }
}

// Select Equation
void 
Math_Main::SelectEqu(int Equ)
{
  if ( ( Equ <= 0) || ( Equ > (int)Equations.count() ) ) {
    Equ = 0;
  }
  else {
    if ( Equations.at(Equ-1) == NULL ) Equ=0;
  };
  if ( ( CurrentEqu > 0 ) && ( Equ == 0 ) ) {
    grey();
   };
   if ( ( CurrentEqu == 0 ) && ( Equ > 0 ) ) {
     ungrey();
   };
  CurrentEqu = Equ;
  Update();
}

// Lookup table for error codes - defined in errors.h file
bool
Math_Main::ErrorCheck(error_type Error)
{
   switch (Error)
   {
   case NO_ERROR :
     return true;
   default :
     QMessageBox::message("Error!",error_msgs[Error]);
     break;
   };
   return false;
}

// Lookup table for error codes - defined in errors.h file
bool
Math_Main::ErrorCheck(error_struct Error)
{
  QString text;

   switch (Error.err_type)
   {
   case NO_ERROR :
     return true;
   default :
     if (Error.extra) {
       text.sprintf("%s : %s",error_msgs[Error.err_type], Error.extra);
       QMessageBox::message("Error!", text);
     }
     else {
       QMessageBox::message("Error!", error_msgs[Error.err_type]);
     }
     if (Error.del)
       delete Error.extra;
     break;
   };
   return false;
}


// Update the contents of the equation labels
void 
Math_Main::Update()
{
  EqnFrame->changeLabels();
  EqnFrame->selectLabel(CurrentEqu);
  EqnFrame->repaint();
}

// Alter the window labels for a resize event
void 
Math_Main::resizeEvent( QResizeEvent * )
{
  int w,h;
  h = height();
  w = width();
  w -= 60;
  outlabel->setGeometry( 30, h-70, w ,20 );
  output->setGeometry( 30, h-50, w , 30 );
  EqnFrame->resize( w, h-180 );
}

// Put a number in the output label
void 
Math_Main::NumberToOutput(double answer)
{
  QString Ans;

  if ( SciNot ) {
    Ans.setNum( answer, 'E', SigFig - 1 );
  }
  else {
    Ans.setNum( answer, 'G', SigFig );
  };
  output->setText( Ans );
  if (LogStream) {
    *LogStream << " =";
    *LogStream << Ans;
    *LogStream << "\n";
  }
  output->repaint();
}

// Copy contents of output label to clipboard
void 
Math_Main::CopyOutput()
{
  clip->setText( output->text() );
}

// Draw a graph of the currently selected equation
void 
Math_Main::DrawGraph(int var)
{
  error_struct err;
  int i;
  double from,step,ans1,ans2;
  double *data;
  Equation *eqn1,*eqn2;
  const int length = 200;
  variable *v1,*v2;
  bool singular = false;
  GraphWin *newgraph;

  v1 = NULL;
  v2 = NULL;

  try {
    data = new double[2*(length+1)];
    from = GraphType.varmin;
    step = (GraphType.varmax-from)/length;
    if (GraphType.Cartesian) {
      if (GraphType.Parametric) {        // Cartesian parametric
	eqn1 = Equations.at(CurrentEqu-1);
	eqn2 = Equations.at(Equ2-1);
	if (var > 0) {
	  v1 = eqn1->Variables.at(var-1);
	  v2 = eqn2->Variables.first();
	  if (v2)
	    while ((v2 != eqn2->Variables.getLast()) && (strcmp(v1->Name,v2->Name)!=0)) v2 = eqn2->Variables.next();
	}
	for (i=0;i<=length;i++) {
	  if (v1) v1->Value = from + i*step;
	  if (v2) v2->Value = v1->Value;
	  try {
	    data[2*i] = eqn1->Evaluate();
	    data[2*i+1] = eqn2->Evaluate();
	  }
	  catch (error_struct err1) {
	    switch (err1.err_type) {
	      // Found a singularity
	    case ERR_DOM: case ERR_RANGE:
	    case ERR_DIV_ZERO:
	      data[2*i] = NAN;
	      data[2*i+1] = NAN;
	      singular = true;
	      break;
	      // Some other error - bail out
	    default:
	      throw err1;
	    }
	  }	  
	}
      }
      else {             // Cartesian non-parametric
	eqn1 = Equations.at(CurrentEqu-1);
	if (var > 0) {
	  v1 = eqn1->Variables.at(var-1);
	}
	for (i=0;i<=length;i++) {
	  data[2*i] = from + i*step;
	  if (v1) v1->Value = data[2*i];
	  try {
	    data[2*i+1] = eqn1->Evaluate();
	  }
	  catch (error_struct err1) {
	    switch (err1.err_type) {
	      // Found a singularity
	    case ERR_DOM: case ERR_RANGE:
	    case ERR_DIV_ZERO:
	      data[2*i+1] = NAN;
	      singular = true;
	      break;
	      // Some other error - bail out
	    default:
	      throw err1;
	    }
	  }
	}
      }
    }
    else {    
      if (GraphType.Parametric) {          // Polar parametric
	eqn1 = Equations.at(CurrentEqu-1);
	eqn2 = Equations.at(Equ2-1);
	if (var > 0) {
	  v1 = eqn1->Variables.at(var-1);
	  v2 = eqn2->Variables.first();
	  if (v2)
	    while ((v2 != eqn2->Variables.getLast()) && (strcmp(v1->Name,v2->Name)!=0)) v2 = eqn2->Variables.next();
	}
	for (i=0;i<=length;i++) {
	  if (v1) v1->Value = from + i*step;
	  if (v2) v2->Value = v1->Value;
	  try {
	    ans1 = eqn1->Evaluate();
	    ans2 = eqn2->Evaluate();
	    data[2*i] = ans1*cos(ans2);
	    data[2*i+1] = ans1*sin(ans2);
	  }
	  catch (error_struct err1) {
	    switch (err1.err_type) {
	      // Found a singularity
	    case ERR_DOM: case ERR_RANGE:
	    case ERR_DIV_ZERO:
	      data[2*i] = NAN;
	      data[2*i+1] = NAN;
	      singular = true;
	      break;
	      // Some other error - bail out
	    default:
	      throw err1;
	    }
	  }
	}      
      }
      else {         // Polar non-parametric
	eqn1 = Equations.at(CurrentEqu-1);
	if (var > 0) {
	  v1 = eqn1->Variables.at(var-1);
	}
	else {
	  QMessageBox::message("Error!","Variable not found for parametric equation");
	  err.err_type = NO_ERROR;
	  throw err;
	}
	for (i=0;i<=length;i++) {
	  v1->Value = from + i*step;
	  try {
	    ans1 = eqn1->Evaluate();
	    data[2*i] = ans1*cos(v1->Value);
	    data[2*i+1] = ans1*sin(v1->Value);
	  }
	  catch (error_struct err1) {
	    switch (err1.err_type) {
	      // Found a singularity
	    case ERR_DOM: case ERR_RANGE:
	    case ERR_DIV_ZERO:
	      data[2*i] = NAN;
	      data[2*i+1] = NAN;
	      singular = true;
	      break;
	      // Some other error - bail out
	    default:
	      throw err1;
	    }
	  }
	}
      }
    }
  }
  catch (error_struct err) {
    ErrorCheck(err);
    if (data != NULL) delete data;
    return;
  }
  if ((GraphType.Replot == true) && (Graph.count())) {
    newgraph = Graph.last();
    newgraph->addPlot(data,length+1,GraphType, singular);
  }
  else {
    newgraph = new GraphWin(this, data, length+1, GraphType, singular, printer);
    newgraph->show();
    connect( newgraph, SIGNAL(deleteGraph(bool, GraphWin *)), SLOT(delete_graph(bool, GraphWin *)) );
    Graph.append(newgraph);
  }
}

// Write out data to file
void 
Math_Main::SaveFile(QString *tempname)
{
  variable *temp;
  Equation *eqn;

  QFile eqnfile(*tempname);
  if (!eqnfile.open(IO_WriteOnly)) {
    QMessageBox::message("Error!","Unable to open file");
    return;
  }
  QDataStream eqnstream(&eqnfile);
  eqnstream.writeRawBytes("mtk",4);
  eqnstream << fileVersion;
  eqnstream << Radians;
  eqnstream << SciNot;
  eqnstream << SigFig;
  eqnstream << Range.From;
  eqnstream << Range.To;
  eqnstream << Range.Step;
  eqnstream << Range.Accuracy;
  eqnstream << GraphType.varmin;
  eqnstream << GraphType.varmax;
  eqnstream << GraphType.xmin;
  eqnstream << GraphType.xmax;
  eqnstream << GraphType.ymin;
  eqnstream << GraphType.ymax;
  eqnstream << GraphType.Cartesian;
  eqnstream << GraphType.Parametric;
  eqnstream << GraphType.Replot;
  eqnstream << GraphType.Auto;
  eqnstream << GraphType.x_tic_space;
  eqnstream << GraphType.y_tic_space;
  eqnstream << GraphType.x_grid_space;
  eqnstream << GraphType.y_grid_space;
  eqnstream << CurrentEqu;
  eqnstream << Equ2;
  eqnstream << Equations.count();
  eqn = Equations.first();
  while (eqn != NULL) {
    eqnstream << eqn->EquationText;
    temp = eqn->Variables.first();
    while (temp != NULL) {
      eqnstream << temp->Value;
      temp = eqn->Variables.next();;
    }
    eqn = Equations.next();
  }
  eqnfile.close();
  Filename = tempname;
}

// Read in data from file
int 
Math_Main::OpenFile(QString *tempname)
{
  int i,Num,NewSel;
  char *text;
  char magic[4];
  variable *temp;
  Equation *eqn;
  uint version;

  QFile eqnfile(*tempname);
  if (!eqnfile.open(IO_ReadOnly)) return -1;
  QDataStream eqnstream(&eqnfile);
  eqnstream.readRawBytes(magic,4);
  if ( strncmp("mtk",magic,4) != 0 ) {
    eqnfile.close();
    return -3;
  }
  eqnstream >> version;
  if ((version > fileVersion) || !version) {
    eqnfile.close();
    return -4;
  }
  eqnstream >> Radians;
  eqnstream >> SciNot;
  eqnstream >> SigFig;
  eqnstream >> Range.From;
  eqnstream >> Range.To;
  eqnstream >> Range.Step;
  eqnstream >> Range.Accuracy;
  eqnstream >> GraphType.varmin;
  eqnstream >> GraphType.varmax;
  if (version < 2) {
    GraphType.xmin = GraphType.varmin;
    GraphType.xmax = GraphType.varmax;
  }
  else {
    eqnstream >> GraphType.xmin;
    eqnstream >> GraphType.xmax;
  }
  eqnstream >> GraphType.ymin;
  eqnstream >> GraphType.ymax;
  eqnstream >> GraphType.Cartesian;
  eqnstream >> GraphType.Parametric;
  eqnstream >> GraphType.Replot;
  eqnstream >> GraphType.Auto;
  if (version < 2) {
    GraphType.x_tic_space = 0;
    GraphType.y_tic_space = 0;
    GraphType.x_grid_space = -1;
    GraphType.y_grid_space = -1;
  }
  else {
    eqnstream >> GraphType.x_tic_space;
    eqnstream >> GraphType.y_tic_space;
    eqnstream >> GraphType.x_grid_space;
    eqnstream >> GraphType.y_grid_space;
  }
  eqnstream >> NewSel;
  eqnstream >> Equ2;
  eqnstream >> Num;  
  Equations.clear();
  for (i=0;i<Num;i++) {
    eqnstream >> text;
    eqn = parser.TextToTree(text);
    delete text;
    eqn->Radians = Radians;
    temp = eqn->Variables.first();
    while (temp != NULL) {
      eqnstream >> temp->Value;
      temp = eqn->Variables.next();
    }
    Equations.append(eqn);
  }
  if ( eqnfile.status() != IO_Ok ) {
    eqnfile.close();
    return -2;
  }
  eqnfile.close();
  Filename = tempname;
  delete_graphs();
  CurrentEqu = 0;
  SelectEqu(NewSel);
  Update();
  if (Num) {
    equationMenu->setItemEnabled(SelectID,TRUE);
    equationMenu->setItemEnabled(DeleteID,TRUE);
    equationMenu->setItemEnabled(DeleteAllID,TRUE);
  }
  return 0;
}

#include "mainwin.moc"
