/*
 * mtk - Maths Toolkit for X11
 *
 * Copyright 1994-1997   andrewr@chiark.greenend.org.uk (Andrew Ross)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

//Class for Graph window

#include <qpainter.h>
#include <qmsgbox.h>
#include <qpdevmet.h>
#include <qkeycode.h>
#include <qstring.h>
#include <qpushbt.h>

#include <math.h>
// Needed for OS/2 port
#ifdef __EMX__
#include <float.h>
#else
#include <nan.h>
#endif

#include "graphwin.h"
#include "mainwin.h"

const QColor plotColour[5]={red,blue,green,yellow,cyan};

GraphWin::GraphWin(Math_Main *parent, double *data, int n, graphsize g, bool singular ,QPrinter *p, const char *name) : QWidget(NULL,name)
{
  Parent = parent;
  printer = p;

  AutoRange = g.Auto;
  IsSingular = singular;
  numplots = 0;
  addPlot(data,n,g,singular);
  
  setFocusPolicy( QWidget::StrongFocus );

  // Set up menu
  graphMenu = new QPopupMenu();
  DeleteID = graphMenu->insertItem("Delete graph", this, SLOT(delete_pressed()), Key_Delete );
  PrintID = graphMenu->insertItem("Print graph", this, SLOT(printPlot()), CTRL + Key_P );

  menu = new QMenuBar(this);
  menu->insertItem("Graph", graphMenu);

}

GraphWin::~GraphWin()
{
  int i;
  
  for (i=0;i<numplots;i++) {
    delete[] Array[i];
  }
  delete graphMenu;
  delete menu;
}

// Add another plot to an existing graph
void 
GraphWin::addPlot(double *d, int n, graphsize g, bool singular)
{
  Equation *eqn,*eqn2;

  if (singular)
    IsSingular = true;

  if (numplots==NUMPLOTS) {
    QMessageBox::message("Error!","Too many plots already");
    return;
  }
  GraphType[numplots] = g;
  Array[numplots] = d;
  ArraySize[numplots] = n;
  Points[numplots].resize(n);
  eqn = Parent->Equations.at(Parent->CurrentEqu-1);
  eqn2 = Parent->Equations.at(Parent->Equ2-1);
  if (GraphType[numplots].Cartesian) {
    if (GraphType[numplots].Parametric) {
      Title[numplots].sprintf("x = %s, y = %s",eqn->EquationText,eqn2->EquationText);
    }
    else {
      Title[numplots].sprintf("y = %s",eqn->EquationText);
    }
  }
  else {
    if (GraphType[numplots].Parametric) {
      Title[numplots].sprintf("r = %s, theta = %s",eqn->EquationText,eqn2->EquationText);
    }
    else {
      Title[numplots].sprintf("r = %s",eqn->EquationText);
    }
  }
  numplots++;
  if (numplots == 1) {
    setRange();
    resize(800,500);
  }
  else {
    resizeEvent(NULL);
  }
  repaint(TRUE);
}

// Draw first plot without deleting and recreating class
void
GraphWin::addFirstPlot(double *d, int n, graphsize g, bool a, bool singular)
{
  int i;
  
  for (i=0;i<numplots;i++) {
    delete[] Array[i];
  }
  AutoRange = a;  
  IsSingular = singular;
  numplots = 0;
  addPlot(d,n,g, singular);
}

// Draw graph with given painter
void
GraphWin::drawGraph( QPainter *paint, bool isScreen )
{
  int i;
  uint j,size;
  QPoint p;
  QPen pen1 = paint->pen();
  QPen pen2 = paint->pen();
  bool nan = false;

  pen2.setWidth(2);

  if (isScreen) {
    paint->setClipRect(GraphRect);
    paint->setPen(black);
  }

  // Draw axes
  paint->setPen(pen2);
  paint->drawLine(Xaxis_min, Xaxis_max);
  paint->drawLine(Yaxis_min, Yaxis_max);
  // Tic marks
  paint->setPen(pen1);
  for (i=0;i<num_xtics + num_ytics;i++) {
    paint->drawLine(Tics.point(i*2), Tics.point(i*2+1));
  }
  // Grid
  for (i=0;i<num_xgrids + num_ygrids;i++) {
    paint->drawLine(Grid.point(i*2), Grid.point(i*2+1));
  }
  // Label axes
  if (isScreen) paint->setClipRect(0,0,width(),height());
  paint->drawText(Xlabel_min,Xmin_text);
  paint->drawText(Xlabel_max,Xmax_text);
  paint->drawText(Ylabel_min,Ymin_text);
  paint->drawText(Ylabel_max,Ymax_text);
  for (i=0;i<numplots;i++) {
    if (isScreen) paint->setPen(plotColour[i%5]);
    paint->drawText(TextRect[i], AlignCenter, Title[i]);
  }

  // Plot graphs
  for (i=0;i<numplots;i++) {
    if (isScreen) paint->setPen(plotColour[i%5]);
    if (IsSingular) {
      size = Points[i].size();
      for (j=0; j<size; j++) {
	p = Points[i].at(j);
	// Last point was NAN
	if (nan) {
	  if (p.y() == QCOORD_MAX) {
	    nan = true;
	  }
	  else {
	    nan = false;
	    paint->moveTo(p);
	  }

	}
	// Last point OK.
	else {
	  if (p.y() == QCOORD_MAX) {
	    nan = true;
	  }
	  else {
	    if (j==0) 
	      paint->moveTo(p);
	    else 
	      paint->lineTo(p);
	  }
	}
      }
    }
    else
      paint->drawPolyline(Points[i]);
  }
}

// Print the plot out
void
GraphWin::printPlot()
{
  QPainter paint;
  int h,w,sh,sw;
  double factor;
  
  if ( printer->setup() ) {
    paint.begin(printer);
    //    paint.setFont( QFont("times",24) );

    QPaintDeviceMetrics pm(printer);
    h = pm.height();
    w = pm.width();
    sh = height();
    sw = width();

    factor = ( ((double)sh)/((double)h) <? ((double)sw)/((double)w) );
    paint.scale(factor,factor);

    drawGraph( &paint, FALSE );

    paint.end();
  }
}

// Set the ranges for the graph
void
GraphWin::setRange()
{
  int i;

  QPainter paint;
  paint.begin(this);
  QFontMetrics font = paint.fontMetrics();

  if (AutoRange) {
    xmin = ymin = DBL_MAX;
    xmax = ymax = -DBL_MAX;  
    for (i = 0; i < 2*ArraySize[0]; i += 2) {
      if (Array[0][i+1] != NAN) {
	if (Array[0][i] > xmax) xmax = Array[0][i];
	if (Array[0][i] < xmin) xmin = Array[0][i];
	if (Array[0][i+1] > ymax) ymax = Array[0][i+1];
	if (Array[0][i+1] < ymin) ymin = Array[0][i+1];
      }
    }
    if ((xmin == DBL_MAX) && (xmax == -DBL_MAX)) {
      xmax = xmin = 0;
      ymax = ymin = 0;
    }
    
    if (xmax == xmin) {
      xmin = xmin - 0.5;
      xmax = xmax + 0.5;
    }
    if (ymax == ymin) {
      ymin = ymin - 0.5;
      ymax = ymax + 0.5;
    }
  }
  else {
    xmin = GraphType[0].xmin;
    xmax = GraphType[0].xmax;
    ymin = GraphType[0].ymin;
    ymax = GraphType[0].ymax;
  }

  Xmin_text.sprintf("%.3G",xmin);
  Xmax_text.sprintf("%.3G",xmax);
  Ymin_text.sprintf("%.3G",ymin);
  Ymax_text.sprintf("%.3G",ymax);
  fontheight = font.height();
  Xmin_shift.setX(-font.width(Xmin_text)/2);
  Xmin_shift.setY(fontheight);
  Xmax_shift.setX(-font.width(Xmax_text)/2);
  Xmax_shift.setY(fontheight);
  Ymin_shift.setX(-font.width(Ymin_text)-2);
  Ymin_shift.setY(fontheight/2);
  Ymax_shift.setX(-font.width(Ymax_text)-2);
  Ymax_shift.setY(fontheight/2);

  xtic_spacing = GraphType[0].x_tic_space;
  ytic_spacing = GraphType[0].y_tic_space;

  setSpacing(xmin, xmax, xtic_spacing, xtic_min, num_xtics);
  setSpacing(ymin, ymax, ytic_spacing, ytic_min, num_ytics);
  Tics.resize((num_xtics+num_ytics)*2);

  xgrid_spacing = GraphType[0].x_grid_space;
  ygrid_spacing = GraphType[0].y_grid_space;

  if (xgrid_spacing < 0)
    num_xgrids = 0;
  else
    setSpacing(xmin, xmax, xgrid_spacing, xgrid_min, num_xgrids);

  if (ygrid_spacing < 0)
    num_ygrids = 0;
  else
    setSpacing(ymin, ymax, ygrid_spacing, ygrid_min, num_ygrids);

  if (num_xgrids || num_ygrids) {
    Grid.resize((num_xgrids + num_ygrids)*2);
  }

  paint.end();
}

// Set the spacing for tic marks or grid
void
GraphWin::setSpacing(double from, double to, double &spacing, double &start, int &number)
{

  if (spacing == 0) {
    // Auto spacing set so calculate appropriate
    spacing = pow(10, floor(log10(to-from) ) - 1 );
  }
  else {
    // Check this is a sensible value
    if ((spacing > to - from) || (spacing < 0))
      spacing = to - from;
  }

  // Need to calculate the position of the first tic/grid mark.
  start = spacing*( (int)(from/spacing) );

  number = (int) ((to - start)/spacing + 1);
}

// Respond to a paint event by drawing graph
void
GraphWin::paintEvent(QPaintEvent *)
{
  QPainter paint;

  paint.begin(this);
  drawGraph( &paint, TRUE );
  paint.end();
}

// Respond to delete button being pressed
void
GraphWin::delete_pressed()
{
  emit deleteGraph(true, this);
}

// Respond to close event
void
GraphWin::closeEvent( QCloseEvent * )
{
  emit deleteGraph(false, this);
}


// Respond to a resize event by rescaling points
void
GraphWin::resizeEvent( QResizeEvent *)
{
  int i,j,x,y,h,w, menuheight,th,tw;
  double xwin,ywin,wleft,wright,htop,hbot;

  h = height();
  w = width();
  menuheight = menu->height();
  htop = h*0.05 + fontheight*1.1*numplots + menuheight;
  hbot = h*0.95;
  wleft = w*0.05;
  wright = w*0.95;
  xwin = (wright-wleft)/(xmax-xmin);
  ywin = (hbot-htop)/(ymax-ymin);
  th = h/100;
  tw = w/100;
  
  for (j=0;j<numplots;j++) {
    for(i=0; i<ArraySize[j]; i++) {
      x = (int)(wleft + (Array[j][2*i]-xmin)*xwin);
      if ( isnan(Array[j][2*i+1]) )
	y = QCOORD_MAX;
      else
	y = (int)(hbot - (Array[j][2*i+1]-ymin)*ywin);
      Points[j].setPoint(i,x,y);
    }
  }
   
  x = (int)(wleft);
  if (ymin>=0) {
    y = (int)(hbot);
  }
  else {
    if (ymax<=0) {
      y = (int)(htop);
    }
    else {
      y = (int)((hbot*ymax - htop*ymin)/(ymax-ymin));
    }
  }
  Xaxis_min.setX(x);
  Xaxis_min.setY(y);
  x = (int)(wright);
  Xaxis_max.setX(x);
  Xaxis_max.setY(y);

  for(i=0; i<num_xtics; i++) {
    x = (int)(wleft + (xtic_min-xmin + i*xtic_spacing)*xwin);
    Tics.setPoint(i*2,x,y-th);
    Tics.setPoint(i*2+1,x,y+th);
  }

  y = (int)(hbot);
  if (xmin>=0) {
    x = (int)(wleft);
  }
  else {
    if (xmax<=0) {
      x = (int)(wright);
    }
    else {
      x = (int)((wleft*xmax - wright*xmin)/(xmax-xmin));
    }
  }
  Yaxis_min.setX(x);
  Yaxis_min.setY(y);
  y = (int)(htop);
  Yaxis_max.setX(x);
  Yaxis_max.setY(y);

  for(i=0; i<num_ytics; i++) {
    y = (int)(hbot - (ytic_min-ymin + i*ytic_spacing)*ywin);
    Tics.setPoint((i+num_xtics)*2,x-tw,y);
    Tics.setPoint((i+num_xtics)*2+1,x+tw,y);
  }

  for (i=0;i<num_xgrids; i++) {
    x = (int)(wleft + (xgrid_min-xmin + i*xgrid_spacing)*xwin);
    Grid.setPoint(i*2, x, (int) hbot);
    Grid.setPoint(i*2+1, x, (int) htop);
  }

  for (i=0;i<num_ygrids; i++) {
    y = (int)(hbot - (ygrid_min-ymin + i*ygrid_spacing)*ywin);
    Grid.setPoint((i+num_xgrids)*2, (int) wleft, y);
    Grid.setPoint((i+num_xgrids)*2+1, (int) wright, y);
  }

  Xlabel_min = Xaxis_min + Xmin_shift;
  Xlabel_max = Xaxis_max + Xmax_shift;
  Ylabel_min = Yaxis_min + Ymin_shift;
  Ylabel_max = Yaxis_max + Ymax_shift; 

  for (i=0;i<numplots;i++) {
    TextRect[i].setLeft( (int)(wleft) );
    TextRect[i].setRight( (int)(wright) );
    TextRect[i].setTop( (int)(h*0.0125 + i*1.1*fontheight +  menuheight) );
    TextRect[i].setBottom( (int)(h*0.0125 + (i+1)*1.1*fontheight +  menuheight) );
  }

  GraphRect.setLeft( (int)(wleft) );
  GraphRect.setRight( (int)(wright) );
  GraphRect.setTop( (int)(htop) );
  GraphRect.setBottom( (int)(hbot) );
}

#include "graphwin.moc"


