/*
 * mtk - Maths Toolkit for X11
 *
 * Copyright 1994-1997   andrewr@chiark.greenend.org.uk (Andrew Ross)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

// Parser class

#define PARSER_CPP 1

#include <math.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include "moremath.h"
#include "parser.h"
#include "equ.h"
#include "func.h"
#include "functext.h"
#include "errors.h"

//Constructor for the Parser class
Parser::Parser()
{
  EquationTokenStart = NULL;
  EquationTokenEnd = NULL;
  EquationText = NULL;
  equation = NULL;
}

//Destructor for the Parser class
Parser::~Parser()
{
  while (EquationTokenStart!=EquationTokenEnd) {
    EquationTokenStart=EquationTokenStart->Next;
    delete EquationTokenStart->Last;
  };
  if (EquationTokenStart!=NULL) delete EquationTokenStart;
}

// Convert text to a tree and return error code
Equation *
Parser::TextToTree( char *text )
{
  error_struct err;
  err.extra = NULL;
  err.del = false;

  try {
    equation = new Equation();

    EquationText = new char[strlen(text) + 1];
    strcpy(EquationText,text);
    if (strlen(EquationText) == 0) {
      err.err_type = ERR_NO_TEXT;
      throw err;
    };
    PassOne();
    ErrorCheck();
    PassTwo();
    RemoveBrackets();
    equation->DumpEqn();
    TidyUp();
    equation->DumpEqn();
    TreeToText();
  }
  catch (...) {
    if (equation) 
    {
      delete equation;
      equation = NULL;
    }
    if (EquationText) {
      delete EquationText;
      EquationText = NULL;
    }
    while (EquationTokenStart!=EquationTokenEnd) {
      EquationTokenStart=EquationTokenStart->Next;
      delete EquationTokenStart->Last;
    };
    if (EquationTokenStart!=NULL) {
      delete EquationTokenStart;
      EquationTokenStart = NULL;
      EquationTokenEnd = NULL;
    }
    throw;
  }

  equation->EquationText = EquationText;
  EquationText = NULL;
  while (EquationTokenStart!=EquationTokenEnd) {
    EquationTokenStart=EquationTokenStart->Next;
    delete EquationTokenStart->Last;
  };
  if (EquationTokenStart!=NULL) {
    delete EquationTokenStart;
    EquationTokenStart = NULL;
    EquationTokenEnd = NULL;
  }
  
  return equation;
}

// Convert tree to text
void
Parser::TreeToText( Equation *eqn )
{
  equation = eqn;

  try {
    TidyUp();
    TreeToText();
  }
  catch (...) {
    if (EquationText) {
      delete EquationText;
      EquationText = NULL;
    }
  }
  
  if (equation->EquationText != NULL) delete equation->EquationText;
  equation->EquationText = EquationText;
  EquationText = NULL;
}

void 
Parser::TidyUpEqn(Equation *eqn)
{
  equation = eqn;

  TidyUp();
  eqn->DumpEqn();
}

// First pass of the parser
void 
Parser::PassOne()
{
  char * Pointer;
  Pointer = EquationText;
  char * ThisFunc = new char[10];
  ThisFunc[0] = 0;
  double * ThisDigit = new double;
  *ThisDigit = 0;
  bool * DoDigit = new bool;
  *DoDigit = false;
  int * DecPt = new int;
  *DecPt = 0;
  equation->brackets = 0;
  
  while ( *Pointer != 0 ) {
    if ( isdigit(*Pointer) ) {
      if ( strlen(ThisFunc) != 0 ) EndFunc(ThisFunc);
      Digit( Pointer, ThisDigit, DoDigit, DecPt );
    }
    if ( isalpha(*Pointer) ) {
      if ( *DoDigit == true ) {      //End of number
	EndNum( ThisDigit, DoDigit, DecPt );
      }
      Alpha( Pointer, ThisFunc );
    }
    if ( ispunct(*Pointer) ) {
      if ( ( *DoDigit == true ) && ( *Pointer != '.' ) ) {      //Not Decimal Pt, end of number
	EndNum( ThisDigit, DoDigit, DecPt );
      }
      if ( strlen(ThisFunc) != 0 ) EndFunc(ThisFunc);
      if ( *Pointer == '.' ) {     //Decimal Pt in number
	*DecPt = 1;
	if ( *DoDigit == false ) {
	  *DoDigit = true;
	  *ThisDigit = 0;
	}
	     }
      else {
	Symbol(Pointer);
      }
    }
    if (isspace(*Pointer)) {
      if ( *DoDigit == true ) EndNum( ThisDigit, DoDigit, DecPt );
      if ( ThisFunc[0] != 0 ) EndFunc( ThisFunc );
    }
    Pointer++;
  }
  if ( *DoDigit == true ) EndNum( ThisDigit, DoDigit, DecPt );
  if ( ThisFunc[0] != 0 ) EndFunc( ThisFunc );
  delete ThisDigit;
  delete DoDigit;
  delete DecPt;
  delete ThisFunc;  
}

// Add next digit to number
void 
Parser::Digit(char * Pointer, double * ThisDigit, bool * DoDigit, int * DecPt)
{
   if ( *DecPt == 0 ) {
      *ThisDigit = (*ThisDigit) * 10 + (*Pointer) - 48;
   }
   else {
     (*ThisDigit) = (*ThisDigit) + ((*Pointer) - 48) / pow( 10, (double)(*DecPt) );
     (*DecPt)++;
   }
   if ( *DoDigit == false ) *DoDigit = true;
}

// Add alphabetic character to this function string
void 
Parser::Alpha(char * Pointer, char * ThisFunc)
{
  if ( strlen(ThisFunc) < 10 ) {
    strncat( ThisFunc, Pointer, 1 );
  }
}

// Check to see what a symbol is
void 
Parser::Symbol(char * Pointer)
{
  error_struct err;

  func_type WhichSymbol = NOFUNC;
  switch (*Pointer) {
  case '!' : 
    WhichSymbol=FACTORIAL;   // !
    break;
  case '(' : WhichSymbol=LBRACKET;    // (
    equation->brackets++;
    break;
  case ')' : WhichSymbol=RBRACKET;    // )
    equation->brackets--;
    break;
  case '*' : WhichSymbol=MULTIPLY;    // *
    break;
  case '+' : WhichSymbol=ADD;         // +
    break;
  case ',': WhichSymbol = COMMA;      // ,
    break;
  case '-' : WhichSymbol=SUBTRACT;    // -
    break;
  case '/' : WhichSymbol=DIVIDE;      // /
    break;
  case '^' : WhichSymbol=POWER;       // ^
    break;
  default :         // Anything else not recognised
    // Tidy up then throw an error
    err.err_type = ERR_SYMBOL;
    err.extra = new char[2];
    err.extra[0] = *Pointer;
    err.extra[1] = '\0';
    err.del = true;
    throw err;
    break;
  };
  if (WhichSymbol!=0) {
    if (EquationTokenEnd && (WhichSymbol == MULTIPLY) && (EquationTokenEnd->Type == MULTIPLY)) {
      EquationTokenEnd->Type = POWER2;
    }
    else {
      AddToken(WhichSymbol);
    }
    (EquationTokenEnd->Value).Number=0;
  };
}

// Check to see if function string is a function
void 
Parser::EndFunc(char * ThisFunc)
{
  func_type WhichFunc=NOFUNC, i;

  if (strcmp(ThisFunc,"C")==0) {
    WhichFunc=COMBINATION;
  }
  else {
    if (strcmp(ThisFunc,"P")==0) {
      WhichFunc=PERMUTATION;
    }
    else {
      if (strcmp(ThisFunc,"pi")==0) WhichFunc=PI_SYM;
      else {
	Alpha(" ",ThisFunc);
	i = SQRT;
	while ( WhichFunc==NOFUNC && i <= FRAC ) {
	  if ( strcmp(ThisFunc,func_details[i].func_text)==0 ) WhichFunc = i;
	  i = i + 1;
	}
      }
    }
  }

  if (WhichFunc != NOFUNC) {
    AddToken(WhichFunc);
    EquationTokenEnd->Value.Number=0;
  }
  else {
    variable *VarPoint;
    VarPoint = equation->Variables.first();
    AddToken(VARIABLE);
    ThisFunc[ strlen(ThisFunc) - 1 ] = 0;
    while (VarPoint!=NULL) {
      if (strcmp(ThisFunc,VarPoint->Name)==0) {
	EquationTokenEnd->Value.WhichVar=equation->Variables.at() + 1;
	ThisFunc[0]=0;
	return;
      }
      VarPoint=equation->Variables.next();
    }
    VarPoint = new variable;
    equation->Variables.append(VarPoint);
    EquationTokenEnd->Value.WhichVar=equation->Variables.count();
    strncpy( VarPoint->Name, ThisFunc, 12 );
    VarPoint->Value = 0;
  }
  ThisFunc[0] = 0;
}

// End of number reached - add token
void 
Parser::EndNum(double * ThisDigit, bool * DoDigit, int * DecPt)
{
    AddToken(NUMBER);
    (EquationTokenEnd->Value).Number=*ThisDigit;
    *ThisDigit=0;
    *DoDigit=false;
    *DecPt=0;
}

// Add a token to the token representation
void 
Parser::AddToken(func_type WhichType)
{
    if (EquationTokenEnd!=NULL)
    {
       EquationTokenEnd->Next=new Token;
       (EquationTokenEnd->Next)->Last=EquationTokenEnd;
       EquationTokenEnd=EquationTokenEnd->Next;
       EquationTokenEnd->Next=NULL;
       EquationTokenEnd->Type=WhichType;
    }
    else
    {
       EquationTokenStart=new Token;
       EquationTokenEnd=EquationTokenStart;
       EquationTokenStart->Last=NULL;
       EquationTokenStart->Next=NULL;
       EquationTokenStart->Type=WhichType;
    };
}



// Check for various errors
void 
Parser::ErrorCheck()
{
  error_struct err;
  err.extra = NULL;
  err.del = false;

  Token *Pointer,*Temp;

  if (equation->brackets != 0) {
    err.err_type = ERR_BRACKET;
    throw err;
  }

  Pointer=EquationTokenStart;
  if (EquationTokenStart==EquationTokenEnd) {
    if ( EquationTokenStart==NULL ) {
      err.err_type = ERR_NO_TEXT;
      throw err;
      return;
    }
    if (((Pointer->Type)<NUMBER)||((Pointer->Type)>PI_SYM)) {
      err.err_type = ERR_FUNCTION;
      err.extra = func_details[Pointer->Type].func_text;
      throw err;
      return;
    };
    return;
  };
  switch (Pointer->Type) {
  case ADD :     
    if ((((Pointer->Next)->Type)>=MULTIPLY)&&(((Pointer->Next)->Type)<=POWER2) || (Pointer->Next->Type == BESSJ)) {
      err.err_type = ERR_OPERATOR;
      err.extra = func_details[ADD].func_text;
      err.del = false;
      throw err;
    };
    (Pointer->Next)->Last=NULL;
    Temp=Pointer;
    Pointer=Pointer->Next;
    delete Temp;
    EquationTokenStart=Pointer;
    break;
  case SUBTRACT :
    if ((((Pointer->Next)->Type)>=MULTIPLY)&&(((Pointer->Next)->Type)<=POWER2) || (Pointer->Next->Type == BESSJ)) {
      err.err_type = ERR_OPERATOR;
      err.extra = func_details[Pointer->Type].func_text;      
      err.del = false;
      throw err;
    };
    if ((Pointer->Next)->Type==NUMBER) {
      ((Pointer->Next->Value).Number)= -((Pointer->Next->Value).Number);
      Pointer->Next->Last=NULL;
      Temp=Pointer;
      Pointer=Pointer->Next;
      delete Temp;
      EquationTokenStart=Pointer;
    }
    else {
      Pointer->Type=NUMBER;
      (Pointer->Value).Number=-1;
      Temp=Pointer->Next;
      Pointer->Next = new Token;
      Pointer->Next->Last = Pointer;
      Pointer->Next->Next = Temp;
      Pointer->Next->Type = MULTIPLY;
      Temp->Last = Pointer->Next;
    };
    break;
  case MULTIPLY : case DIVIDE : case POWER: case POWER2 :
    err.err_type = ERR_OPERATOR;
    err.extra = func_details[Pointer->Type].func_text;      
    err.del = false;
    throw err;
  case COMBINATION : case PERMUTATION : case FACTORIAL : case COMMA :
    err.err_type = ERR_FUNCTION;
    err.extra = func_details[Pointer->Type].func_text;      
    err.del = false;
    throw err;
  case RBRACKET :
    err.err_type = ERR_BRACKET;
    throw err;
  default:
    break;
  };

  while ((Pointer->Next!=EquationTokenEnd)&&(Pointer->Next!=NULL)) {
    Pointer=Pointer->Next;
    switch (Pointer->Type) {
    case MULTIPLY : case DIVIDE : case POWER : case POWER2 : 
      if ( (Pointer->Last->Type != LBRACKET) && ((Pointer->Last->Type < FACTORIAL) || (Pointer->Last->Type > RBRACKET)) ) {
	err.err_type = ERR_OPERATOR;
	err.extra = func_details[Pointer->Type].func_text;      
	err.del = false;
	throw err;
      };
      break;

    case FACTORIAL : case COMBINATION : case PERMUTATION : case COMMA :
      if ( (Pointer->Last->Type < NUMBER) || (Pointer->Last->Type > RBRACKET) ) {
	if (Pointer->Last->Type != INT) {
	  err.err_type = ERR_FUNCTION;
	  err.extra = func_details[Pointer->Type].func_text;      
	  err.del = false;
	  throw err;
	}
      }
      break;
    case ADD : case SUBTRACT :
      if ( (Pointer->Next->Type >= MULTIPLY) && (Pointer->Next->Type <= POWER2) || (Pointer->Next->Type == COMMA)) {
	err.err_type = ERR_OPERATOR;
	err.extra = func_details[Pointer->Type].func_text;      
	err.del = false;
	throw err;
      };
      if ( ( (Pointer->Last->Type < NUMBER) || (Pointer->Last->Type > RBRACKET) ) && (Pointer->Last->Type != FACTORIAL) ){
	if ( (Pointer->Next->Type == NUMBER) && (Pointer->Type == SUBTRACT) ) 
	  Pointer->Next->Value.Number = - Pointer->Next->Value.Number;
	if ( (Pointer->Next->Type == NUMBER) || (Pointer->Type == ADD) ) {
	  Pointer->Next->Last = Pointer->Last;
	  Temp = Pointer;
	  Pointer = Pointer->Next;
	  delete Temp;
	  Pointer->Last->Next = Pointer;
	}
	else {
	  Pointer->Type=NUMBER;
	  (Pointer->Value).Number=-1;
	  Temp=Pointer->Next;
	  Pointer->Next = new Token;
	  Pointer->Next->Last = Pointer;
	  Pointer->Next->Next = Temp;
	  Pointer->Next->Type = MULTIPLY;
	  Temp->Last = Pointer->Next;
	};
      };
      break;

    case RBRACKET :
      if ((((Pointer->Last)->Type)<LBRACKET)
	  || (((Pointer->Last)->Type)>RBRACKET)) {
	err.err_type = ERR_BRACKET;
	throw err;
      };
      break;

    default :
      if ((Pointer->Last->Type > LBRACKET) && (Pointer->Last->Type < COMMA)) {
	Temp = Pointer->Last;
	Pointer->Last=new Token;
	Pointer->Last->Next=Pointer;
	Pointer=Pointer->Last;
	Pointer->Last=Temp;
	Temp->Next=Pointer;
	Pointer->Type=MULTIPLY;
	Pointer->Value.Number=0;
	Pointer=Pointer->Next;
      };
    };
  };
  Pointer=Pointer->Next;
  if (Pointer) {
    if ((((Pointer->Type)>=COMBINATION)&&((Pointer->Type)<=FRAC)) 
	|| (Pointer->Type == COMMA) ) {
      err.err_type = ERR_FUNCTION;
      err.extra = func_details[Pointer->Type].func_text;      
      err.del = false;
      throw err;
    };
    if ((Pointer->Type)==LBRACKET) {
      err.err_type = ERR_BRACKET;
      throw err;
    };
    if (((Pointer->Type)<=POWER2) && ((Pointer->Type)>=ADD)) {
      err.err_type = ERR_OPERATOR;
      err.extra = func_details[Pointer->Type].func_text;      
      err.del = false;
      throw err;
    };
    if (((Pointer->Type)>=NUMBER)&&((Pointer->Type)<=PI_SYM)) {
      if ((Pointer->Last->Type >= NUMBER) && (Pointer->Last->Type < COMMA)) {
	Temp=Pointer->Last;
	Pointer->Last=new Token;
	Pointer->Last->Next=Pointer;
	Pointer=Pointer->Last;
	Pointer->Last=Temp;
	Temp->Next=Pointer;
	Pointer->Type=MULTIPLY;
	Pointer->Value.Number=0;
	Pointer=Pointer->Next;
      }
    }
  }
}


// Second pass of equation parser
void 
Parser::PassTwo()
{
   Node *TreePos;
   Token *Pointer;
   error_struct err;
   err.extra = NULL;

   Pointer = EquationTokenStart;
   TreePos = equation->Root;
   TreePos->Type=Pointer->Type;
   TreePos->evalfn = func_details[TreePos->Type].func_ptr;
   TreePos->Value = Pointer->Value;

   while (Pointer->Next!=NULL) {
     Pointer=Pointer->Next;

     if (Priority(TreePos->Type,Pointer->Type)==true) {
       if ((TreePos->Parent!=NULL)&&(Pointer->Type!=FACTORIAL)) {
	 while (Priority((TreePos->Parent)->Type,Pointer->Type)==true) {
	   TreePos=TreePos->Parent;
	   if (TreePos->Parent==NULL) break;
	 }
       }
       if (Pointer->Type==RBRACKET) {
	 if (TreePos->Parent==NULL) {
	   err.err_type = ERR_BRACKET;
	   throw err;
	 }
	 TreePos=TreePos->Parent;
       }
       TreePos=equation->AddParent(TreePos);
     }
     else {
       if ((((TreePos->Type>=ADD)&&(TreePos->Type)<=POWER2))||(Pointer->Type==LBRACKET) || (TreePos->Type == COMMA)) {
	 if (Pointer->Type==LBRACKET) {
	   if (TreePos->Child_One==NULL) {
	     TreePos=equation->AddChildOne(TreePos);
	   }
	   else {
	     TreePos=equation->AddChildTwo(TreePos);
	   }
	 }
	 else {
	   TreePos=equation->AddChildTwo(TreePos);
	 }
       }
       else {
	 while ((Priority(TreePos->Type,Pointer->Type)!=true)
		||(TreePos->Child_One!=NULL)) {
	   if (TreePos->Child_One==NULL) break;
	   if (TreePos->Child_Two==NULL) {
	     TreePos=TreePos->Child_One;
	   }
	   else {
	     TreePos=TreePos->Child_Two;
	   }
	 }
	 if (Priority(TreePos->Type,Pointer->Type)==true) {
	   TreePos=equation->AddParent(TreePos);
	 }
	 else {
	   TreePos=equation->AddChildOne(TreePos);
	 }
       }
     }
     equation->InitNode(TreePos,Pointer->Type,Pointer->Value);
     TreePos->Child_Two=NULL;
   }
}

// Returns true if First is higher or equal priority than Second 
// else returns false
bool 
Parser::Priority(func_type First, func_type Second)         
{
    if (Second==LBRACKET) return false;
    if (First==LBRACKET) return false;
    if (First==RBRACKET) return true;
    if (Second==RBRACKET) return true;
    if (Second == COMMA) return true;
    if (First == COMMA) return false;
    if ((First==ADD)||(First==SUBTRACT))
    {
       if ((Second==ADD)||(Second==SUBTRACT)) return true;
       return false;
    };
    if ((First==MULTIPLY)||(First==DIVIDE))
    {
       if ((Second>=ADD)&&(Second<=DIVIDE)) return true;
       return false;
    };
    if (((First == POWER) || (First == POWER2)) && ((Second == POWER) || (Second == POWER2))) return true;
    if (Second==FACTORIAL) return true;
    if ((Second>=ADD)&&(Second<=DIVIDE)) return true;
    if (Second == BESSJ) return true;
    if ((First>=NUMBER)&&(First<=RBRACKET)) return true;
    return false;
}

// Returns true if First is higher or equal priority than Second 
// else returns false
// This version is used for converting tree to text
bool 
Parser::TextPriority(func_type First, func_type Second, bool childtwo)         
{
    if (Second==LBRACKET) return false;
    if (First==LBRACKET) return false;
    if (First==RBRACKET) return true;
    if (Second==RBRACKET) return true;
    if (Second == COMMA) return true;
    if (First == COMMA) return false;
    if ((First == ADD) || (First == SUBTRACT))
    {
       if ((Second == ADD) || ((Second == SUBTRACT) && !childtwo)) return true;
       return false;
    };
    if ((First == MULTIPLY) || (First==DIVIDE))
    {
       if ((Second >= ADD) && (Second <= MULTIPLY)) return true;
       if ((Second == DIVIDE) && !childtwo) return true;
       return false;
    };
    if (Second==FACTORIAL) return true;
    if ((Second>=ADD)&&(Second<=DIVIDE)) return true;
    if (Second == BESSJ) return true;
    if ((First>=NUMBER)&&(First<=RBRACKET)) return true;
    return false;
}


// Remove unnecessary brackets from the equation
void 
Parser::RemoveBrackets()
{
  error_struct err;
  Node *TreePos, *temp;

  err.extra = NULL;
  err.del = false;

   TreePos = equation->Root;
   while (equation->Root->Brackets==0) {
     while (TreePos->Child_One!=NULL) {
       if ((TreePos->Child_One)->Brackets==1) break;
       TreePos=TreePos->Child_One;
     };
     if ((TreePos->Type==LBRACKET)||(TreePos->Type==RBRACKET)) {
       TreePos=equation->DeleteNode(TreePos);
       if (TreePos==NULL) break;
     }
     else {
       if (TreePos->Child_Two==NULL) {
	 TreePos->Brackets=1;
	 TreePos=TreePos->Parent;
       }
       else {
	 if ((TreePos->Child_Two)->Brackets==1) {
	   TreePos->Brackets=1;
	   TreePos=TreePos->Parent;
	 }
	 else {
	   TreePos=TreePos->Child_Two;
	 }
       }
     }
   }
   if ((equation->Root->Type==LBRACKET)||(equation->Root->Type==RBRACKET)) {
     equation->Delete(equation->Root);
   }

   equation->Reset(0);
   TreePos = equation->Root;
   while (equation->Root->Brackets==0) {
     while (TreePos->Child_One!=NULL) {
       if ((TreePos->Child_One)->Brackets==1) break;
       TreePos=TreePos->Child_One;
     };
     if (TreePos->Type == COMMA) {
       if ( !TreePos->Parent || !TreePos->Child_One || !TreePos->Child_Two 
	    || (TreePos->Parent->Type != BESSJ)) {
	 err.err_type = ERR_FUNCTION;
	 err.extra = func_details[BESSJ].func_text;
	 throw err;
       }
       temp = TreePos;
       TreePos = TreePos->Parent;
       TreePos->Child_One = temp->Child_One;
       TreePos->Child_Two = temp->Child_Two;
       temp->Child_One->Parent = TreePos; 
       temp->Child_Two->Parent = TreePos;
       delete temp;
     }
     else {
       if (TreePos->Child_Two==NULL) {
	 TreePos->Brackets=1;
	 TreePos=TreePos->Parent;
       }
       else {
	 if (TreePos->Child_Two->Brackets == 1) {
	   TreePos->Brackets=1;
	   TreePos=TreePos->Parent;
	 }
	 else {
	   TreePos=TreePos->Child_Two;
	 }
       }
     }
   }
}


// Tidy up equation tree
void 
Parser::TidyUp()
{
  Node *TreePos;

  TreePos = equation->Root;
  DoSubDiv(TreePos);

  GroupNumbers(MULTIPLY);

  equation->Reset(0);
  while ( equation->Root->Brackets == 0 ) {
    while ( TreePos->Child_One != NULL ) {
      if ( TreePos->Child_One->Brackets == 1 ) break;
      TreePos = TreePos->Child_One;
    };
    TreePos->Brackets = 1;
    TreePos = TreePos->Parent;
    if ( TreePos == NULL ) break;
    if ( TreePos->Child_Two == NULL ) {
      TidyNode(TreePos);
    }
    else {
      if ( TreePos->Child_Two->Brackets == 0 ) {
	TreePos = TreePos->Child_Two;
      }
      else {
	TidyNode( TreePos );
      }
    }
  }
  
  GroupNumbers(ADD);

  TreePos = equation->Root;
  equation->Reset(0);
  while ( equation->Root->Brackets == 0 ) {
    while ( TreePos->Child_One != NULL ) {
      if ( TreePos->Child_One->Brackets == 1 ) break;
      TreePos=TreePos->Child_One;
    };
    TreePos->Brackets = 1;
    TreePos = TreePos->Parent;
    if ( TreePos == NULL ) break;
    if ( TreePos->Child_Two == NULL ) {
      TidyNode(TreePos);
    }
    else {
      if ( (TreePos->Child_Two)->Brackets==0 ) {
	TreePos=TreePos->Child_Two;
      }
      else {
	TidyNode( TreePos );
      }
    }
  }
}


// Tidy up this node
void 
Parser::TidyNode(Node *&TreePos)
{
  error_struct err;
  Node *Temp;
  double value;
  NumVar tmpnum;

  err.extra = NULL;

  switch (TreePos->Type) {
  case ADD: case SUBTRACT:
    if ((TreePos->Child_One->Type == NUMBER)&&(TreePos->Child_Two->Type == NUMBER)) {
      if (TreePos->Type == ADD) {
	TreePos->Value.Number = TreePos->Child_One->Value.Number + TreePos->Child_Two->Value.Number;
      }
      else {
	TreePos->Value.Number = TreePos->Child_One->Value.Number - TreePos->Child_Two->Value.Number;
      }
      TreePos->Type = NUMBER;
      TreePos->evalfn=func_details[NUMBER].func_ptr;
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One = NULL;
      TreePos->Child_Two = NULL;
      break;
    }
    if ((TreePos->Child_One->Type == NUMBER) && ((TreePos->Child_One->Value).Number == 0)) {
      if (TreePos->Type == ADD) {
	TreePos = equation->DeleteNode(TreePos,TreePos->Child_One,TreePos->Child_Two);
      }
      else {
	equation->InitNode(TreePos, MULTIPLY);
	tmpnum.Number = -1;
	equation->InitNode(TreePos->Child_One, NUMBER, tmpnum);
      }
      break;
    }
    if ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number == 0)) {
      TreePos = equation->DeleteNode(TreePos,TreePos->Child_Two,TreePos->Child_One);
      break;
    };
    if ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number < 0)) {
      TreePos->Child_Two->Value.Number = -(TreePos->Child_Two->Value.Number);
      if (TreePos->Type == ADD) TreePos->Type = SUBTRACT;
      else TreePos->Type = ADD;
      TreePos->evalfn=func_details[TreePos->Type].func_ptr;
      break;
    };
    if ((TreePos->Child_Two->Child_One) && (TreePos->Child_Two->Child_One->Type == NUMBER) && ((TreePos->Child_Two->Child_One->Value).Number < 0)) {
      TreePos->Child_Two->Child_One->Value.Number = -(TreePos->Child_Two->Child_One->Value.Number);
      if (TreePos->Type == ADD) TreePos->Type = SUBTRACT;
      else TreePos->Type = ADD;
      TreePos->evalfn=func_details[TreePos->Type].func_ptr;
      break;
    };
    break;

  case MULTIPLY:
    if ((TreePos->Child_One->Type == NUMBER) && (TreePos->Child_Two->Type == NUMBER)) {
      TreePos->Value.Number = TreePos->Child_One->Value.Number * TreePos->Child_Two->Value.Number;
      TreePos->Type = NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One = NULL;
      TreePos->Child_Two = NULL;
      break;
    }
    if (((TreePos->Child_One->Type == NUMBER) && ((TreePos->Child_One->Value).Number == 0)) 
	|| ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number == 0))) {
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Type=NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      (TreePos->Value).Number=0;
      TreePos->Child_One=NULL;
      TreePos->Child_Two=NULL;
      break;
    };
    if ((TreePos->Child_One->Type == NUMBER) && ((TreePos->Child_One->Value).Number == 1)) {
      TreePos = equation->DeleteNode(TreePos,TreePos->Child_One,TreePos->Child_Two);
      break;
    };
    if ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number == 1)) {
      TreePos = equation->DeleteNode(TreePos,TreePos->Child_Two,TreePos->Child_One);
      break;
    };
    if ((TreePos->Child_One->Type >= SQRT) && (TreePos->Child_One->Type <= FRAC)) {
      if ((TreePos->Child_Two->Type < SQRT) || (TreePos->Child_Two->Type > FRAC)) {
        Temp=TreePos->Child_One;
        TreePos->Child_One=TreePos->Child_Two;
        TreePos->Child_Two=Temp;
      };
    };
    break;

  case DIVIDE:
    if ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number == 0)) {
      err.err_type = ERR_DIV_ZERO;
      throw err;
    };
    if ((TreePos->Child_One->Type == NUMBER) && (TreePos->Child_Two->Type == NUMBER)) {
      TreePos->Value.Number = TreePos->Child_One->Value.Number / TreePos->Child_Two->Value.Number;
      TreePos->Type = NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One = NULL;
      TreePos->Child_Two = NULL;
      break;
    }
    if ((TreePos->Child_One->Type == NUMBER) && ((TreePos->Child_One->Value).Number == 0)) {
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Type=NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      (TreePos->Value).Number=0;
      TreePos->Child_One=NULL;
      TreePos->Child_Two=NULL;
      break;
    }
    if ((TreePos->Child_Two->Type == NUMBER) && ((TreePos->Child_Two->Value).Number == 1)) {
      TreePos = equation->DeleteNode(TreePos,TreePos->Child_Two,TreePos->Child_One);
      break;
    }
    break;
  case POWER : case POWER2 : 
    if (((TreePos->Child_One)->Type==NUMBER)&&(((TreePos->Child_One)->Value).Number==0)
	&&((TreePos->Child_Two)->Type==NUMBER)&&(((TreePos->Child_Two)->Value).Number==0)) {
      err.err_type = ERR_DIV_ZERO;
      throw err;
    };
    if (((TreePos->Child_One)->Type==NUMBER)&&(((TreePos->Child_One)->Value).Number==0)) {
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One=NULL;
      TreePos->Child_Two=NULL;
      TreePos->Type=NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      (TreePos->Value).Number=0;
      break;
    }
    if (((TreePos->Child_Two)->Type==NUMBER)&&(((TreePos->Child_Two)->Value).Number==0)) {
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One=NULL;
      TreePos->Child_Two=NULL;
      TreePos->Type=NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      (TreePos->Value).Number=1;
      break;
    }
    if (((TreePos->Child_Two)->Type==NUMBER)&&(((TreePos->Child_Two)->Value).Number==1)) {
      TreePos = equation->DeleteNode(TreePos, TreePos->Child_Two, TreePos->Child_One);
      break;
    }
    if (((TreePos->Child_One)->Type==NUMBER)&&((TreePos->Child_Two)->Type==NUMBER)) {
      TreePos->Value.Number = pow(TreePos->Child_One->Value.Number,TreePos->Child_Two->Value.Number);
      TreePos->Type = NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      equation->Delete(TreePos->Child_One);
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_One = NULL;
      TreePos->Child_Two = NULL;
      break;
    }
    break;

  case BESSJ:
  case COMBINATION:
  case PERMUTATION:
    if ((TreePos->Child_One->Type == NUMBER) && (TreePos->Child_Two->Type == NUMBER)) {
      switch (TreePos->Type) {
      case BESSJ:
	value = jn((int)TreePos->Child_One->Value.Number,TreePos->Child_Two->Value.Number);
	break;
      case COMBINATION:
	value = combination(TreePos->Child_One->Value.Number, TreePos->Child_Two->Value.Number);
	break;
      case PERMUTATION:
	value = permutation(TreePos->Child_One->Value.Number, TreePos->Child_Two->Value.Number);
	break;
      default:
	break;
      }
      TreePos->Type = NUMBER;
      TreePos->evalfn = func_details[NUMBER].func_ptr;
      TreePos->Value.Number = value;
      equation->Delete(TreePos->Child_One);
      TreePos->Child_One = NULL;
      equation->Delete(TreePos->Child_Two);
      TreePos->Child_Two = NULL;
    }
    break;
  default :
    if ( ( (TreePos->Type) >= SQRT ) && ( (TreePos->Type) <= FACTORIAL) ) {
      if (TreePos->Child_One->Type == NUMBER) {
	if ( equation->Radians || (TreePos->Type < SIN) || ( TreePos->Type > ARCTAN) ) {
	  value = (*(TreePos->evalfn))(TreePos->Child_One->Value.Number);
	}
	else {
	  if ( TreePos->Type <= COT ) 
	    value = (*(TreePos->evalfn))(TreePos->Child_One->Value.Number*DEGTORAD);
	  else
	    value = (*(TreePos->evalfn))(TreePos->Child_One->Value.Number)/DEGTORAD;
	  }
	TreePos->Type = NUMBER;
	TreePos->evalfn = func_details[NUMBER].func_ptr;
	TreePos->Value.Number = value;
	equation->Delete(TreePos->Child_One);
	TreePos->Child_One = NULL;
      }
    }
  }
  
  TreePos->Brackets=1;
}

// This iterates over the tree and groups all the numbers together. 
// It operates on expressions linked by type Type which
// is MULTIPLY or ADD
// e.g. x*y*4*sin x
//      5+2x+sin y
void
Parser::GroupNumbers(func_type Type)
{
  Node *TreePos, *temp;
  TreePos = equation->Root;

  equation->Reset(0);
  while ( equation->Root->Brackets == 0 ) {
    while ( TreePos->Child_One != NULL ) {
      if ( TreePos->Child_One->Brackets == 1 ) break;
      TreePos=TreePos->Child_One;
    };
    TreePos->Brackets = 1;
    TreePos = TreePos->Parent;
    if ( TreePos == NULL ) break;
    if ( TreePos->Child_Two != NULL ) {
      if ( TreePos->Child_Two->Brackets == 0 ) {
	TreePos = TreePos->Child_Two;
      }
      else {
	if ((TreePos->Type == Type) && ( ((TreePos->Parent) && (TreePos->Parent->Type != Type)) || !(TreePos->Parent))) {
	  // Rearrange the series of * or + signs
	  Rearrange( TreePos, Type );
	  // Evaluate any series of numbers now
	  temp = NULL;
	  if (TreePos->Child_Two->Type == NUMBER) {
	    temp = TreePos;
	  }
	  else {
	    if (TreePos->Child_One->Type == Type) {
	      temp = TreePos->Child_One;
	    }
	  }
	  if (temp) {
	    temp->Value.Number = equation->EvalFromNode(temp);
	    temp->Type = NUMBER;
	    temp->evalfn = NULL;
	    equation->Delete(temp->Child_One);
	    equation->Delete(temp->Child_Two);
	    temp->Child_One = NULL;
	    temp->Child_Two = NULL;
	  }
	}
      }
    }
  }
}


// Rearrange a set of multiplications signs so all numbers are on the left
// branch and everything else on the right
void
Parser::Rearrange(Node *TreeRoot, func_type Type)
{
  Node *LeftPos, *RightPos, *Temp, *NewPos;

  TreeRoot->Brackets = 1;
  equation->Reset(0, TreeRoot);
  // root must be * sign else return
  if (TreeRoot->Type != Type)
    return;

  LeftPos = TreeRoot->Child_One;
  RightPos = TreeRoot->Child_Two;
  
  while (TRUE) {
    // Get next pointers in left and right branches
    // When all done then pointer is NULL
    LeftPos = GetNextPos(LeftPos, TreeRoot, false, Type );
    RightPos = GetNextPos(RightPos, TreeRoot, true, Type);

    // We've finished - both pointers are NULL
    if ( !LeftPos && !RightPos ) {
      TreeRoot->Brackets = 1;
      return;
    }


    if (LeftPos) {
      //
      // We've got pointers on each side so swap them
      //
      if (RightPos) {
	if (LeftPos->Parent == RightPos->Parent) {
	  // Share the same parent
	  LeftPos->Parent->Child_One = RightPos;
	  RightPos->Parent->Child_Two = LeftPos;
	}
	else {
	// Swap parents over
	  Temp = LeftPos->Parent;
	  LeftPos->Parent = RightPos->Parent;
	  RightPos->Parent = Temp;
	  //Correct parent's children
	  if (RightPos->Parent->Child_One == LeftPos)
	    RightPos->Parent->Child_One = RightPos;
	  else
	    RightPos->Parent->Child_Two = RightPos;
	  if (LeftPos->Parent->Child_One == RightPos)
	    LeftPos->Parent->Child_One = LeftPos;
	  else
	    LeftPos->Parent->Child_Two = LeftPos;
	}
	// Exchange left and right pointers to reflect changes
	Temp = LeftPos;
	LeftPos = RightPos;
	RightPos = Temp;
      }
      else {
	//
	// We've got LeftPos but not RightPos
	//
	if (LeftPos->Parent != TreeRoot) {
	  Temp = LeftPos->Parent;
	  NewPos = Temp->Parent;
	  if (Temp->Child_One == LeftPos) {
	    // It's a child one
	    Temp->Child_Two->Parent = Temp->Parent;
	    if (Temp->Parent->Child_One == Temp) {
	      Temp->Parent->Child_One = Temp->Child_Two;
	    }
	    else {
	      Temp->Parent->Child_Two = Temp->Child_Two;
	    }
	    // Move left pointer to child two for consistency
	    Temp->Child_Two = LeftPos;
	  }
	  else {
	    // It's a child two
	    Temp->Child_One->Parent = Temp->Parent;
	    if (Temp->Parent->Child_One == Temp) {
	      Temp->Parent->Child_One = Temp->Child_One;
	    }
	    else {
	      Temp->Parent->Child_Two = Temp->Child_One;
	    }
	  }
	  Temp->Child_One = TreeRoot->Child_Two;
	  Temp->Parent = TreeRoot;
	  TreeRoot->Child_Two = Temp;
	  Temp->Child_One->Parent = Temp;
	  LeftPos = NewPos;
	}
	else {
	  LeftPos = TreeRoot;
	}
      }
    }	
    else{
      //
      // We've got RightPos but not LeftPos
      //
      if (RightPos->Parent != TreeRoot) {
	Temp = RightPos->Parent;
	NewPos = Temp->Parent;
	if (Temp->Child_One == RightPos) {
	  // It's a child one
	  Temp->Child_Two->Parent = Temp->Parent;
	  if (Temp->Parent->Child_One == Temp) {
	    Temp->Parent->Child_One = Temp->Child_Two;
	    }
	  else {
	    Temp->Parent->Child_Two = Temp->Child_Two;
	  }
	  // Move right pointer to child two for consistency
	  Temp->Child_Two = RightPos;
	}
	else {
	  // It's a child two
	  Temp->Child_One->Parent = Temp->Parent;
	  if (Temp->Parent->Child_One == Temp) {
	    Temp->Parent->Child_One = Temp->Child_One;
	  }
	  else {
	    Temp->Parent->Child_Two = Temp->Child_One;
	  }
	}
	Temp->Child_One = TreeRoot->Child_One;
	Temp->Parent = TreeRoot;
	TreeRoot->Child_One = Temp;
	Temp->Child_One->Parent = Temp;
	RightPos = NewPos;
      }
      else {
	RightPos = TreeRoot;
      }
    }
  }
}

// Return next position iterating over tree with root TreeRoot
// If numbers is true then looking for numbers else looking for
// not numbers
// When the tree is done return NULL
Node *
Parser::GetNextPos( Node *TreePos, Node *TreeRoot, bool numbers, func_type Type )
{
  // Already covered all of the tree
  if (!TreePos)
    return NULL;

  // If already at the bottom
  if (!TreePos->Child_One) {
    TreePos->Brackets = 1;
    if (numbers && (TreePos->Type == NUMBER))
      return TreePos;
    if (!numbers && !(TreePos->Type == NUMBER))
      return TreePos;
  }

  // If we're at an already covered node
  if (TreePos->Brackets == 1)
    TreePos = TreePos->Parent;

  while (TreePos != TreeRoot) {
    // Go down child one's until we find a node not of type Type
    if (TreePos->Child_One && (TreePos->Child_One->Brackets == 0)) {
      while (TreePos->Type == Type) {
	if (TreePos->Child_One->Brackets == 1) 
	  break;
	TreePos = TreePos->Child_One;
      }
      TreePos->Brackets = 1;
      if (numbers && (TreePos->Type == NUMBER))
	return TreePos;
      if (!numbers && !(TreePos->Type == NUMBER))
	return TreePos;
      TreePos = TreePos->Parent;
    }
    else {
      // Done child one so do child two
      if (TreePos->Child_Two && (TreePos->Child_Two->Brackets == 0)) {
	TreePos = TreePos->Child_Two;
	if (!TreePos->Child_One) {
	  TreePos->Brackets = 1;
	  if (numbers && (TreePos->Type == NUMBER))
	    return TreePos;
	  if (!numbers && !(TreePos->Type == NUMBER))
	    return TreePos;
	  TreePos = TreePos->Parent;
	}
      }
      // Done child two as well so go up one
      else {
	TreePos->Brackets = 1;
	TreePos = TreePos->Parent;
      }
    }
  }
      
  return NULL;
}

// Remove multiple subtract and divide signs
void
Parser::DoSubDiv(Node *TreePos)
{
  Node *Temp;

  // Check not null pointer
  if (TreePos == NULL)
    return;

  // No children so return
  if (TreePos->Child_One == NULL)
    return;

  // Do children first
  DoSubDiv(TreePos->Child_One);
  if (TreePos->Child_Two) 
    DoSubDiv(TreePos->Child_Two);
  
  // Only interested in subtract and divide types
  if ((TreePos->Type != DIVIDE) && (TreePos->Type != SUBTRACT) && (TreePos->Type != POWER) && (TreePos->Type != POWER2)) 
    return;

  /*  if ((TreePos->Type == ADD) && (TreePos->Child_One) && (TreePos->Child_One->Type == SUBTRACT)) {
    Temp = TreePos;
    TreePos->Child_One->Parent = TreePos->Parent;
    if (TreePos->Parent) {
      if (TreePos->Parent->Child_One == TreePos) {
        TreePos->Parent->Child_One;
      }
    }
  }*/

  // Check if there is anything of the form x-(y-z)
  // If so rearrange as x-y+z
  if ((TreePos->Child_Two->Type == TreePos->Type) && (TreePos->Type != POWER) && (TreePos->Type != POWER2)) {
    if (TreePos->Type == SUBTRACT) 
      TreePos->Type = ADD;
    else
      TreePos->Type = MULTIPLY;
    Temp = TreePos->Child_Two->Child_One;
    TreePos->Child_Two->Child_One = TreePos->Child_Two->Child_Two;
    TreePos->Child_Two->Child_Two = Temp;
  }
  else {
    // Rearrange to eliminate one minus or divide sign
    // E.g. x-y-z = x-(y+z)
    // Also copes with x^y^z = x^(y*z)
    if (TreePos->Child_One->Type == TreePos->Type) {
      Temp = TreePos->Child_Two;
      TreePos->Child_Two  = TreePos->Child_One;
      TreePos->Child_One = TreePos->Child_Two->Child_One;
      TreePos->Child_One->Parent = TreePos;
      TreePos->Child_Two->Child_One = TreePos->Child_Two->Child_Two;
      TreePos->Child_Two->Child_Two = Temp;
      Temp->Parent = TreePos->Child_Two;
      if (TreePos->Type == SUBTRACT)
	TreePos->Child_Two->Type = ADD;
      else
	TreePos->Child_Two->Type = MULTIPLY;
    }
  }
}
    
// Convert tree to text
void 
Parser::TreeToText()
{
  int Length;
  char *Buffer;
  
  equation->Reset(0);
  Buffer = TreeToText(equation->Root);

  Length=strlen(Buffer);
  EquationText=new char[Length+1];
  strcpy(EquationText,Buffer);
  delete Buffer;
}

// Convert tree from node to text
char *
Parser::TreeToText(Node *TreeRoot)
{
  error_struct err;
  bool Flag=true;
  int length, extra;
  Node *TreePos;
  char *Buffer, *ch1, *ch2, *tempbuffer;
  Buffer = new char[1024];
  int BufferSize = 1023;
  TreePos = TreeRoot;
  Buffer[0] = 0;
  err.extra = NULL;
  err.del = false;
  
  while (Flag) {  
    while (TreePos->Child_One != NULL){
      if (TreePos->Child_One->Brackets == 1) break;
      if ((TreePos->Type >= SQRT) && (TreePos->Type <= FRAC)) {   
	NodeText(TreePos->Type,TreePos,Buffer, BufferSize);
      }
      if (TreePos->Type == BESSJ) {
	if (!TreePos->Child_One || !TreePos->Child_Two) {
	  err.err_type = ERR_FUNCTION;
	  err.extra = func_details[BESSJ].func_text;
	  throw err;
	}
	ch1 = TreeToText(TreePos->Child_One);
	ch2 = TreeToText(TreePos->Child_Two);
	length = strlen(ch1) + strlen(ch2) + 9;
	extra = BufferSize - strlen(Buffer);
	if (extra < length) {
	  tempbuffer = Buffer;
	  Buffer = new char[BufferSize + extra + 100];
	  strcpy(Buffer, tempbuffer);
	  delete tempbuffer;
	}
	strcat(Buffer,"( ");
	strcat(Buffer,ch1);
	strcat(Buffer," , ");
	strcat(Buffer,ch2);
	strcat(Buffer," ) ");
	delete ch1;
	delete ch2;
	TreePos->Brackets = 1;
	break;
      }
      else
	if ( TextPriority(TreePos->Child_One->Type, TreePos->Type, false) == false ) {
	  NodeText(LBRACKET, NULL, Buffer, BufferSize);
	};
      TreePos=TreePos->Child_One;
    };
    if ((TreePos->Type<SQRT)||(TreePos->Type>FRAC)) {
      NodeText(TreePos->Type,TreePos,Buffer, BufferSize);
    }
    if ((TreePos->Child_Two == NULL) || (TreePos->Type == BESSJ)) {
      do {
	if (TreePos != TreeRoot) {
	  if ((TextPriority(TreePos->Type, TreePos->Parent->Type, (TreePos->Parent->Child_Two == TreePos)) == false)
	      && (TreePos->Child_One != NULL)){
	    NodeText(RBRACKET, NULL, Buffer, BufferSize);
	  }
	}
	if (TreePos == TreeRoot) {
	  Flag = false;
	  break;
	};
	TreePos=TreePos->Parent;
      }
      while (TreePos->Brackets==1);
    }
    else {
      if (TextPriority(TreePos->Child_Two->Type,TreePos->Type, true)==false)
	NodeText(LBRACKET, NULL, Buffer, BufferSize);
      TreePos = TreePos->Child_Two;
    }
  }
  return Buffer;
}

// Convert node to text
void 
Parser::NodeText(int type, Node *TreePos,char *Buffer,int &BufferSize)
{
  error_struct err;
  char Temp[30];
  char format[20];
  char *tempbuffer;
  int extra;

  err.extra = NULL;
  err.del = false;

   if ((type <1) || (type>NUMBER_OF_FUNCS)) {
     if (TreePos) 
       TreePos->Brackets=1;
     err.err_type = ERR_FUNCTION;
     err.extra = "Unknown function type";
     throw err;
   }
   if ( (type != NUMBER) && (type != VARIABLE) ) {
     if ( (type != MULTIPLY) || ( (TreePos->Child_One->Type != NUMBER) || (TreePos->Child_Two->Type == NUMBER) ) ) {
	extra = strlen(Buffer) + strlen(func_details[type].func_text) - BufferSize;
	if ( extra >= -1 ) {
	  extra +=100;
	  tempbuffer = Buffer;
	  Buffer = new char[BufferSize + extra];
	  strcpy(Buffer,tempbuffer);
	  delete tempbuffer;
	}
	strcat(Buffer,func_details[type].func_text);
      }
   }
   else {
     if (type == NUMBER){
       if ( (TreePos->Value.Number == -1) && (TreePos->Parent) && (TreePos->Parent->Type == MULTIPLY) && (TreePos->Parent->Child_One == TreePos))
	 sprintf(Temp,"-");
       else {
	 if (SciNot) 
	   sprintf(format, "%%.%dE", SigFig - 1);
	 else
	   sprintf(format, "%%.%dG", SigFig);
	 sprintf(Temp,format,(TreePos->Value).Number);
       }
     }
     else {
       sprintf(Temp,"%s",equation->Variables.at(TreePos->Value.WhichVar - 1)->Name);
     }
     extra = strlen(Buffer) + strlen(Temp) - BufferSize;
     if ( extra >= -1 ) {
       extra +=50;
       tempbuffer = Buffer;
       Buffer = new char[BufferSize + extra];
       strcpy(Buffer,tempbuffer);
       delete tempbuffer;
     }
     strcat(Buffer,Temp);
   }

   if (TreePos) TreePos->Brackets=1;
}









