#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include "gcode.h"


#define DPRINTF 	
/* Program for converting output from the ghostscript pswrite
driver into G-CODE suitable for use for driving a CNC milling
machine.  The sort of application envisaged for this program is
controlling an engraving setup or something of that sort.

(C) January 2003 Alan Bain

The program is basically a simple PS susbet stack based
compiler which reads input and provided it confirms to the
standard (which is fairly like that used in PDF files) it is 
parsed and g-code generated.  It would be simple to modify to
generate some other vector language e.g. HPGL.

WARNING: Use of this code to control a machine is potentially
dangerous.  Please check the CNC cutter path generated using a
suitable tool path program to ensure it is correct and check 
that it clears all work piece clamps etc.

Metal surface is assumed to lie at Z=0.0


ToDO:
 * Handle dashed lines (via setdash) correctly.  At the moment will
   be cut as a solid line. 
 * Handle machine parameters (safe height above work), feed rates
   correctly.  At moment defaults have been entered.
 * Different flatnesses are possible in PS (e.g. type I founts have
   a flatness of 0).  
 * Shallow curves should invoke an extra stage flattening algorithm
 * replace bezier curve computations by an optimized method

*/

#define TRUE 1
#define FALSE 0

#define PUSH(x) stack[++top_stack]=x
#define POP(x)  x=stack[top_stack--]

#define POPALL  top_stack=-1

#define INDEX(x) stack[top_stack-(x)]

#define TOPIND(x) stack[x]

#define SIZE(x)    (x=top_stack+1)

#define MAXLINE 256
#define MAX_STACK 1024

#define max(x,y)  ( (x) > (y) ? (x) : (y) )

// Globals
int penup=FALSE;
int newpath=TRUE;

FILE *file;  // Output file for G-CODE
double path_start_x, path_start_y; // Start of a newpath
                                   // needed to do a closepath
double current_x, current_y;       // current tool position

double stack[MAX_STACK];           // parameter stack
int top_stack=-1;                  // number of top of stack
                                   // start with empty stack

double cut_depth=0.05;             // Depth of cut in inches

double flatness=2;                   // How close a curve should be 
                                   // approximated


double max_x=0.0, max_y=0.0;

void pstack() {
	int i;
	for (i=0; i<=top_stack; i++) 
		printf("%f ",stack[i]);

	printf("\n");
}



void perr(char *mesg){
	fprintf(stderr,"gcode error: %s\n",mesg);
	close(file);
	exit(-1);
}

int main(int argc, char **argv) 
{
	FILE *infile;  // Input Postscript lies in this file
	char line[MAXLINE];

	// Option parsing
	int i=1;
	char *pOption;

	while(i<argc){
		pOption=argv[i];
		if (*pOption == '-')
		{
			pOption++; // Skip -
			if (*pOption=='\0') 
			   perr("Option error - not followed by characters\n");
		}
		else 
		{
			break; // from while
		}

		switch(*pOption)
		{
			case 'd':
			 	cut_depth=strtod(argv[i+1],(char **) NULL);	
				i++;
				break;

			case 'f':
			 	flatness=strtod(argv[i+1], (char **) NULL);	
				i++; // used extra argument
				break;
				
			default:
				perr("Unrecognised Option\n");
		}
		i++;	 // next argument
	} // While	
	printf("i : %d  argc : %d\n",i,argc);

//	argc == 1 IN=STDIN OUT=STDOUT
//      argc == 2 IN= argv[1] OUT=STDOUT
//      argc == 3 IN= argv[1] OUT=argv[2]

	file=stdout;
	infile=stdin;

	
	if (argc>2+i) 
		perr("Too many arguments");
	
	if (argc>i) {
		printf("Opening %s for input\n",argv[i]);
	  	infile=fopen(argv[i],"r");
	}

	if (argc>1+i) {
		printf("Opening %s for output\n",argv[i+1]);
		file=fopen(argv[i+1],"w");
	}
	
	// Look for %%EndPageSetup and start processing from there
	while(!feof(infile)) {
		fgets(line,MAXLINE,infile);
		if (strstr(line,"%%EndPageSetup")!=NULL) 
			{
				
				DPRINTF("Found Key\n");
				break;
			}
	}

	do_preamble();
	while(!feof(infile))
		read_command(infile);
	do_postamble();
	close(file);
	close(infile);

	printf("Max x (inches) %f  Max y (inches) %f\n",xpos(max_x), ypos(max_y));
	// End of Task


}

void curve_to_param(double x_0, double y_0,
     double x_1, double y_1, double x_2, double y_2, double x_3, double y_3,
     params *p)
{
	p->cx=(x_1-x_0)*3.0;
	p->cy=(y_1-y_0)*3.0;

	p->bx=(x_2-x_1)*3.0-p->cx;
	p->by=(y_2-y_1)*3.0-p->cy;

	p->ax=x_3-x_0-p->bx-p->cx;
	p->ay=y_3-y_0-p->by-p->cy;

	p->x_0=x_0;
	p->y_0=y_0;

	p->k = num_points(x_0,y_0,x_1,y_1,x_2,y_2,x_3,y_3);

	DPRINTF("curve: x0 %f y0 %f  ax %f, bx %f, cx %f , ay %f , by %f, cy %f, k %d\n",
		p->x_0, p->y_0, p->ax, p->bx, p->cx, p->ay, p->by, p->cy, p->k);
}


int num_points(double x_0, double y_0, double x_1,
  double y_1, double x_2, double y_2, double x_3, double y_3) 
{	
	// find number of interpolation points needed to
	// approximate the curve accurately enough.  Try
	// for a power of 2 to avoid quantization errors

	// based on algorithm in comments in GS gxpflat.c

	double d,dx,dx2,dy,dy2;
	unsigned int q;
	int k;

	dx=x_0-2.0*x_1+x_2;
	dy=y_0-2.0*y_1+y_2;
	dx2=x_1-2.0*x_2+x_3;
	dy2=y_1-2.0*y_2+y_3;
	d=max(abs(dx)+abs(dy), abs(dx2)+abs(dy2));

	DPRINTF("D is %f\n", d);

	d=d*0.75;
	q=d/flatness;
	
	DPRINTF("q is %d\n", q);

	// Hacky ceil(log2(...)/2) ceiling implementation!

	for (k=0; q>1;)  k++, q=(q+3)>>2;
	return(k);
}


void bezier(double t, params *p)
{
// outputs G-code to lineto point on bezier curve
// with parameter t in [0,1].

// Formula for bezier curve taken from p565 Red Book

// This implementation
// is not particularly efficient.  A better approach is
// to consider the values of f(x+e)-f(x) and use a finite
// differences approach to minimize the number of multiplications

	double x,y;
	double t2,t3;

	t2=t*t;
	t3=t2*t;

	// Parametric cubics now specify curve

	x= p->ax*t3+p->bx*t2+p->cx*t+p->x_0;
	y= p->ay*t3+p->by*t2+p->cy*t+p->y_0;

	fprintf(file,"G1 X%f Y%f\n",xpos(x),ypos(y));

	current_x=x;
	current_y=y;


}

void output_curve(params *p)
{ 
	// Generate g-code output for line segments

	int i,num_step= 1<<p->k;
	double t=0.0,ss;	

	DPRINTF("Number of steps %d\n",num_step);

	
	ss=1.0/num_step;

	do_pen_down();
	for(i=0; i<= num_step; i++,t+=ss)
		bezier(t,p);

}


// To allow us to do arbitrary scaling stuff

double xpos(double x)
{
	return(x/72.0);
}

double ypos(double y)
{
	return(y/72.0);
}


void do_preamble()
{
	fprintf(file,"G20\n");    // Select inches
        fprintf(file,"G17 ");     // X-Y plane
	fprintf(file,"G40 G49 "); // Cancel tool lengh & cutter dia compensation
	fprintf(file,"G53 ");     // Motion in machine co-ordinate system
	fprintf(file,"G80\n");    // Cancel any existing motion cycle

	fprintf(file,"G90\n");    // Absolute distance mode

}


void do_postamble()
{
	do_pen_up();
	fprintf(file,"M5\n"); // Spindle Stop
	fprintf(file,"M2\n"); // End of my program

}

void do_pen_down()
{
	if (penup) {
		fprintf(file,"G0 Z0.1\n");
		fprintf(file,"G1 Z%f\n",-cut_depth);  // Specify a feed rate with
						      // F option
		penup=FALSE;
	}
}

void do_pen_up()
{ 
	if (!penup) {
		fprintf(file,"G1 Z0.1\n");
		fprintf(file,"G0 Z0.5\n");  // Clearance for moves
		penup=TRUE;
	}
}




void do_moveto() {
	double x,y;
	POP(y);
	POP(x);

	DPRINTF("do moveto\n");
	do_pen_up();
	fprintf(file,"G0 X%f Y%f\n",xpos(x),ypos(y));

	current_x=x;
	current_y=y;
	
	//moveto starts a new sub-path of the current path (see p190)
	path_start_x=x;
	path_start_y=y;

	

}

void do_lineto() {
	double x,y;
	POP(y);
	POP(x);

	do_pen_down();
	fprintf(file,"G1 X%f Y%f\n",xpos(x),ypos(y));

	current_x=x;
	current_y=y;

	if (current_x>max_x) max_x=current_x;
	if (current_y>max_y) max_y=current_y;

}

void do_rcurveto() {
	double x1,x2,x3,y1,y2,y3;
	params p;


	POP(y3);
	POP(x3);
	POP(y2);
	POP(x2);
	POP(y1);
	POP(x1);

	DPRINTF("In do_rcurveto x,y %f %f  x1 y1 %f %f x2 y2 %f %f x3 y3 %f %f\n",
		current_x, current_y, x1,y1,x2,y2,x3,y3);

	// Convert to absolute positions
	x1+=current_x;
	x2+=current_x;
	x3+=current_x;

	y1+=current_y;
	y2+=current_y;
	y3+=current_y;


	curve_to_param(current_x,current_y,x1,y1,x2,y2,x3,y3,&p);

	output_curve(&p);

	current_x=x3;
	current_y=y3;

	if (current_x>max_x) max_x=current_x;
	if (current_y>max_y) max_y=current_y;

}

void do_p(int skip) {
	double x,y;
	int pairs,num,stacksz;

	SIZE(stacksz);
	stacksz-=skip;  // Skip some pairs
	pairs=stacksz/2;
	for(num=0; num<pairs; num++){

		x=TOPIND(num*2+skip);
		y=TOPIND(num*2+1+skip);	

	        do_pen_down();

		current_x+=x;
		current_y+=y;
		if (current_x>max_x) max_x=current_x;
		if (current_y>max_y) max_y=current_y;

		fprintf(file,"G1 X%f Y%f\n",xpos(current_x),ypos(current_y));
	}
	POPALL;
}

void do_P() 
{
	int stacksz;
	SIZE(stacksz);
	DPRINTF("do_P\n");
	// Copy bottom onto top of stack
	if (stacksz>0) {
		PUSH(TOPIND(0));
		PUSH(TOPIND(1));

		do_moveto(); // bottom pair on stack
		do_p(2);  // rest (skip first pair);
	}
}

void closepath()
{
	PUSH(path_start_x);
	PUSH(path_start_y);
	do_lineto();
}

// Take a string representing a number and convert to a 
// float

void do_number(char *word) {
	int pos=0;
	double number;
	number=strtod(word,(char **) NULL);
	PUSH(number);
}


void read_command(FILE *infile){
	
	char c[MAXLINE];
	int index=0;

// Grab a word

	while(!isspace (c[index++]=fgetc(infile)) &&!feof(infile) );

	c[--index]='\0'; // Terminate string
	
	DPRINTF("Word: %s length %d\n",c,strlen(c));


	if (c[0]=='.'||c[0]=='-'||isdigit(c[0])) {
	   DPRINTF("\t Number: %s\n",c);
	   do_number(c);
	   return;
	}


// Single character words 
	if (strlen(c)==1) {
		switch(c[0]){

		case 'm': // moveto
			do_moveto();
			break;
		case 'l': // lineto
			do_lineto();
			break;
		
		case 'p':  // multiple rlinetos
			// any number of xy pairs
			do_p(0);
			break;
		case 'P':  // moveto follwed by a p
			do_P();
			break;

		case 'c': // rcurveto (requires real work)
			do_rcurveto();
			break;

		case '^': // take (a,b,c,d) and place (-a, -b) on top
			PUSH(-INDEX(3));
			PUSH(-INDEX(3));
			break;

		case 'Y': // P clip newpath  (newpath is needed to
			  // break the path)

			// We can't do anything with this
			// at present although in some
			// circumstances maybe we should.

			// Hopefully the pswrite phase has removed
			// most of the critical parts of this clip
			// operation.

			// pop all items off stack

			POPALL;
			break;

		case 'S': // P stroke
			do_P();
			break;
			
		case 'f': // P fill [skip fill for moment]
			do_P();
			break;


		case 'h': // p closepath 
			do_p(0);
			closepath();
			break;	

		case 'H': // P closepath
			do_P();
			closepath();
			break;

		// All of the following are commands which should be
		// recognised but which don't correspond immediately
		// to a sensible action to be taken by a milling 
		// machine.  The case statements fall through to
		// the break, but we must remove arguments from the
		// stack appropriately!



		case 'w': // num setlinewidth
		case 'J': // int setlinecap
	 	case 'j': // int setlinejoin
		case 'M': // num setmitrelimit
		case 'd': // array offset setdash
		case 'i': // num setflat

		case 'q': //gsave
		case 'Q': //grestore
		case 'K': // 0 setgray (i.e. choose black lines)
		case 'G': // setgray

		

		POPALL;  // Discard all args from stack  
		break;

		default:
			// Unrecognised command
			printf("Command: %c\n",c[0]);
			perr("Unrecognised command in pswrite output\n");

		}


	return; // Single letter command
	}
}
