/* srcgen.cc
 *
 * Copyright 2009 Martin Read
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define SRCGEN_CC

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include "indie.hh"
#include "pmon1.hh"
#include "pobj1.hh"
#include "srcgen.hh"

char *create_tag_from_name(const char *name, const char *prefix)
{
    int buflen;
    int preflen = strlen(prefix);
    char *buf;
    int i;

    buflen = strlen(name) + preflen + 1;
    buf = new char[buflen];
    sprintf(buf, "%s%s", prefix, name);
    for (i = preflen; buf[i]; i++)
    {
        /* Convert letters to upper case, leave digits unmolested, and
         * turn everything else into underscores. */
        if (isalpha(buf[i]))
        {
            buf[i] = toupper(buf[i]);
        }
        else if (!isdigit(buf[i]))
        {
            buf[i] = '_';
        }
    }
    return buf;
}

namespace
{
    void generate_spiral(void);
    void usage(void);

    enum Edge
    {
        North_edge,
        East_edge,
        South_edge,
        West_edge
    };

    void generate_spiral(void)
    {
        int i;
        FILE *fp;

        fp = fopen("spiralpath.cc", "w");
        if (!fp)
        {
            fprintf(stderr, "Couldn't open spiralpath.cc for write: %s\n", strerror(errno));
            exit(2);
        }
        fprintf(fp, "/* This file is automatically generated. DO NOT EDIT! */\n\n");
        fprintf(fp, "#define SPIRALPATH_CC\n#include \"dunbash.hh\"\n#include \"radiance.hh\"\nlibmrl::Coord spiral_path[21 * 21] = \n{\n");
        int cur_radius = 0;
        int cur_y = 0;
        int cur_x = 0;
        Edge cur_edge = North_edge;
        for (i = 0; i < 21 * 21; ++i)
        {
            fprintf(fp, "{ %d, %d },\n", cur_y, cur_x);
            switch (cur_edge)
            {
            case North_edge:
                if (!cur_radius)
                {
                    --cur_y;
                    ++cur_radius;
                }
                else if (cur_x == cur_radius)
                {
                    cur_edge = East_edge;
                    ++cur_y;
                }
                else
                {
                    ++cur_x;
                    if (cur_x == 0)
                    {
                        ++cur_radius;
                        --cur_y;
                    }
                }
                break;
            case East_edge:
                if (cur_y == cur_radius)
                {
                    cur_edge = South_edge;
                    --cur_x;
                }
                else
                {
                    ++cur_y;
                }
                break;
            case South_edge:
                if (cur_x == -cur_radius)
                {
                    cur_edge = West_edge;
                    --cur_y;
                }
                else
                {
                    --cur_x;
                }
                break;
            case West_edge:
                if (cur_y == -cur_radius)
                {
                    cur_edge = North_edge;
                    ++cur_x;
                    if (cur_x == 0)
                    {
                        ++cur_radius;
                        --cur_y;
                    }
                }
                else
                {
                    --cur_y;
                }
                break;
            }
        }
        fprintf(fp, "};\n");
    }

    void usage(void)
    {
        fprintf(stderr, "Usage:\n\nidgen [-m filename] [-o] [-s]\n");
        fprintf(stderr, "  -m  Generate permon source from text file\n");
        fprintf(stderr, "  -o  Generate permobj source\n");
        fprintf(stderr, "  -s  Generate spiral path\n");
        exit(1);
    }
}

int main(int argc, char *argv[], char *env[])
{
    int ch;
    bool do_pmonid = false;
    bool do_pobjid = false;
    bool do_spiral = false;
    char *permons_fname = 0;
    char *permobj_fname = 0;

    while ((ch = getopt(argc, argv, "o:m:s")) != -1)
    {
        switch (ch)
        {
        case 'm':
            do_pmonid = true;
            permons_fname = strdup(optarg);
            break;

        case 'o':
            do_pobjid = true;
            permobj_fname = strdup(optarg);
            break;

        case 's':
            do_spiral = true;
            break;

        case '?':
            usage();
        }
    }
    if (!do_pmonid && !do_pobjid && !do_spiral)
    {
        usage();
    }
    else
    {
        if (do_pmonid)
        {
            parse_permons(permons_fname);
        }
        if (do_pobjid)
        {
            parse_permobj(permobj_fname);
        }
        if (do_spiral)
        {
            generate_spiral();
        }
    }
    return 0;
}

/* srcgen.cc */
