/* rooms.cc
 * 
 * Copyright 2005-2009 Martin Read
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "dunbash.hh"
#include "rooms.hh"
#include "monsters.hh"
#include "objects.hh"

#include <string.h>

void Levext_rooms::excavate_normal_room(int rnum)
{
    libmrl::Coord topleft = bounds[rnum][0];
    libmrl::Coord botright = bounds[rnum][1];
    libmrl::Coord c;
    for (c.y = topleft.y + 1; c.y < botright.y; c.y++)
    {
        for (c.x = topleft.x + 1; c.x < botright.x; c.x++)
	{
	    parent->set_terrain(c, FLOOR);
	    parent->set_region(c, rnum);
	}
    }
    for (c.y = topleft.y; c.y <= botright.y; c.y++)
    {
        c.x = topleft.x;
        parent->set_region(c, rnum);
        c.x = botright.x;
        parent->set_region(c, rnum);
    }
    for (c.x = topleft.x; c.x <= botright.x; c.x++)
    {
        c.y = topleft.y;
        parent->set_region(c, rnum);
        c.y = botright.y;
        parent->set_region(c, rnum);
    }
}

void Levext_rooms::excavate_shrine(int rnum)
{
    libmrl::Coord topleft = bounds[rnum][0];
    libmrl::Coord botright = bounds[rnum][1];
    libmrl::Coord c = { (topleft.y + botright.y) / 2, (topleft.x + botright.x) / 2 };
    excavate_normal_room(rnum);
    currlev->set_terrain(c, ALTAR);
    switch (zero_die(5))
    {
    case 0:
        currlev->set_terrain(c + dbash::NORTHWEST, LAVA_POOL);
        currlev->set_terrain(c + dbash::NORTHEAST, LAVA_POOL);
        currlev->set_terrain(c + dbash::SOUTHWEST, LAVA_POOL);
        currlev->set_terrain(c + dbash::SOUTHEAST, LAVA_POOL);
        break;
    case 1:
        currlev->set_terrain(c + dbash::NORTHWEST, IRON_FLOOR);
        currlev->set_terrain(c + dbash::NORTHEAST, IRON_FLOOR);
        currlev->set_terrain(c + dbash::SOUTHWEST, IRON_FLOOR);
        currlev->set_terrain(c + dbash::SOUTHEAST, IRON_FLOOR);
        break;
    case 2:
        currlev->set_terrain(c + dbash::NORTHWEST, BONE_FLOOR);
        currlev->set_terrain(c + dbash::NORTHEAST, BONE_FLOOR);
        currlev->set_terrain(c + dbash::SOUTHWEST, BONE_FLOOR);
        currlev->set_terrain(c + dbash::SOUTHEAST, BONE_FLOOR);
        break;
    case 3:
        currlev->set_terrain(c + dbash::NORTHWEST, ACID_POOL);
        currlev->set_terrain(c + dbash::NORTHEAST, ACID_POOL);
        currlev->set_terrain(c + dbash::SOUTHWEST, ACID_POOL);
        currlev->set_terrain(c + dbash::SOUTHEAST, ACID_POOL);
        break;
    case 4:
        currlev->set_terrain(c + dbash::NORTHWEST, WATER_POOL);
        currlev->set_terrain(c + dbash::NORTHEAST, WATER_POOL);
        currlev->set_terrain(c + dbash::SOUTHWEST, WATER_POOL);
        currlev->set_terrain(c + dbash::SOUTHEAST, WATER_POOL);
        break;
    }
}

void Levext_rooms::excavate_morgue(int rnum)
{
    libmrl::Coord topleft = bounds[rnum][0];
    libmrl::Coord botright = bounds[rnum][1];
    libmrl::Coord c;
    excavate_normal_room(rnum);
    for (c.y = topleft.y + 2; c.y < botright - 2; ++(c.y))
    {
        for (c.x = topleft.x + 2; c.x < botright - 2; c.x += 2)
        {
            currlev->set_terrain(c, TOMBSTONE);
        }
    }
}

void Levext_rooms::excavate_smithy(int rnum)
{
    libmrl::Coord topleft = bounds[rnum][0];
    libmrl::Coord botright = bounds[rnum][1];
    libmrl::Coord c = { (topleft.y + botright.y) / 2, (topleft.x + botright.x) / 2 };
    libmrl::Coord c2;
    excavate_normal_room(rnum);
    currlev->set_terrain(c, ANVIL);
    do
    {
        c2 = get_obj_scatter(c + dbash::SOUTHEAST);
    } while ((c.y <= topleft.y) || (c.y >= botright.y) ||
             (c.x <= topleft.x) || (c.x >= botright.x));
    currlev->set_terrain(c2, FURNACE);
}

void Levext_rooms::add_random_room(int yseg, int xseg)
{
    int roomidx = (yseg * 3) + xseg;
    libmrl::Coord centre;
    libmrl::Coord topleft;
    libmrl::Coord botright;
    int ht;
    int wd;
    centre.y = (parent->height - 2) / 6 + yseg * ((parent->height - 2) / 3);
    centre.x = (parent->width - 2) / 6 + xseg * ((parent->width - 2) / 3);
    ht = libmrl::max(6, 2 + dice(2, ROOM_HT_DELTA));
    wd = libmrl::max(5, 2 + dice(2, ROOM_WD_DELTA));
    topleft.y = centre.y - ROOM_HT_DELTA - 1 + zero_die(((ROOM_HT_DELTA + 1) * 2) - ht);
    topleft.x = centre.x - ROOM_WD_DELTA - 1 + zero_die(((ROOM_WD_DELTA + 1) * 2) - wd);
    botright.y = topleft.y + ht;
    botright.x = topleft.x + wd;
    bounds[roomidx][0] = topleft;
    bounds[roomidx][1] = botright;
    segsused[yseg * 3 + xseg] = 1;
    if (roomidx == zoo_room)
    {
        switch (zoo_style)
        {
        case ZOO_MORGUE:
            excavate_morgue(roomidx);
            break;
        case ZOO_SHRINE:
            excavate_shrine(roomidx);
            break;
        case ZOO_TREASURE:
            excavate_normal_room(roomidx);
            break;
        case ZOO_SMITHY:
            excavate_smithy(roomidx);
            break;
        default:
            excavate_normal_room(roomidx);
            break;
        }
    }
    else
    {
        excavate_normal_room(roomidx);
    }
}

void Levext_rooms::link_rooms(int r1, int r2)
{
    libmrl::Coord pos[4];
    int i;
    libmrl::Coord end1;
    libmrl::Coord posts[4];
    libmrl::Coord end2;
    libmrl::Coord mid1;
    libmrl::Coord mid2;
    /* Update the linkage matrix. */
    linkage[r1][r2] = 1;
    linkage[r2][r1] = 1;
    for (i = 0; i < MAX_ROOMS; i++)
    {
	if ((i == r1) || (i == r2))
	{
	    continue;
	}
	if ((linkage[r1][i] > 0) && !linkage[r2][i])
	{
	    linkage[r2][i] = 2;
	    linkage[i][r2] = 2;
	}
	if ((linkage[r2][i] > 0) && !linkage[r1][i])
	{
	    linkage[r1][i] = 2;
	    linkage[i][r1] = 2;
	}
    }
    /* Take a copy of the corners of the rooms, and use them to find an entry
     * and exit point. */
    pos[0] = bounds[r1][0];
    pos[1] = bounds[r1][1];
    pos[2] = bounds[r2][0];
    pos[3] = bounds[r2][1];
    /* Now generate the corridor. */
    if ((r1 % 3) == (r2 % 3))
    {
	if (pos[1].y < pos[2].y)
	{
            end1.y = pos[1].y;
            end2.y = pos[2].y;
	}
	else
	{
            end1.y = pos[0].y;
            end2.y = pos[3].y;
	}
        do
        {
            end1.x = exclusive_flat(pos[0].x, pos[1].x);
        } while (parent->flags_at(end1) & MAPFLAG_NOPIERCE);
        do
        {
            end2.x = exclusive_flat(pos[2].x, pos[3].x);
        } while (parent->flags_at(end2) & MAPFLAG_NOPIERCE);
        mid1.y = exclusive_flat(end1.y, end2.y);
        mid1.x = end1.x;
        mid2.y = mid1.y;
        mid2.x = end2.x;
        posts[0] = end1 + dbash::EAST;
        posts[1] = end1 + dbash::WEST;
        posts[2] = end2 + dbash::EAST;
        posts[3] = end2 + dbash::WEST;
    }
    else
    {
	if (pos[1].x < pos[2].x)
	{
            end1.x = pos[1].x;
            end2.x = pos[2].x;
	}
	else
	{
            end1.x = pos[0].x;
            end2.x = pos[3].x;
	}
        do
        {
            end1.y = exclusive_flat(pos[0].y, pos[1].y);
        } while (parent->flags_at(end1) & MAPFLAG_NOPIERCE);
        do
        {
            end2.y = exclusive_flat(pos[2].y, pos[3].y);
        } while (parent->flags_at(end2) & MAPFLAG_NOPIERCE);
        mid1.x = exclusive_flat(end1.x, end2.x);
        mid1.y = end1.y;
        mid2.x = mid1.x;
        mid2.y = end2.y;
        posts[0] = end1 + dbash::SOUTH;
        posts[1] = end1 + dbash::NORTH;
        posts[2] = end2 + dbash::SOUTH;
        posts[3] = end2 + dbash::NORTH;
    }
    parent->set_flag_at(end1, MAPFLAG_NOPIERCE);
    parent->set_flag_at(end2, MAPFLAG_NOPIERCE);
    parent->set_flag_at(posts[0], MAPFLAG_NOPIERCE);
    parent->set_flag_at(posts[1], MAPFLAG_NOPIERCE);
    parent->set_flag_at(posts[2], MAPFLAG_NOPIERCE);
    parent->set_flag_at(posts[3], MAPFLAG_NOPIERCE);
    excavate_corridor_segment(end1, mid1, zero_die(10), false);
    excavate_corridor_segment(mid1, mid2, false, false);
    excavate_corridor_segment(mid2, end2, false, zero_die(10));
}

void Levext_rooms::put_stairs(void)
{
    dstairs_pos.y = exclusive_flat(bounds[dstairs_room][0].y, bounds[dstairs_room][1].y);
    dstairs_pos.x = exclusive_flat(bounds[dstairs_room][0].x, bounds[dstairs_room][1].x);
    parent->set_terrain(dstairs_pos, STAIRS_DOWN);
    ustairs_pos.y = exclusive_flat(bounds[ustairs_room][0].y, bounds[ustairs_room][1].y);
    ustairs_pos.x = exclusive_flat(bounds[ustairs_room][0].x, bounds[ustairs_room][1].x);
    parent->set_terrain(ustairs_pos, STAIRS_UP);
}

int edge_rooms[4] = { 1, 3, 5, 7 };
int corners[4][2] = { { 0, 2 }, { 0, 6 }, { 2, 8 }, { 6, 8 } };

void Levext_rooms::excavate(void)
{
    int i;
    parent = parent;
    /*
     * For now, stick with the policy that we have nine rooms. This will need
     * rejigging when I get round to making the number of rooms variable.
     */
    actual_rooms = MAX_ROOMS;
    // Select dstairs room, ustairs room, zoo room
    dstairs_room = zero_die(actual_rooms);
    do
    {
        ustairs_room = zero_die(actual_rooms);
    } while (ustairs_room == dstairs_room);
    do
    {
        zoo_room = zero_die(actual_rooms);
    } while ((zoo_room == ustairs_room) ||
             (zoo_room == dstairs_room));
    if (!zero_die(10) && (depth > 2))
    {
        zoo_style = ZOO_TREASURE;
    }
    else if (!zero_die(9) && (depth > 4))
    {
        zoo_style = ZOO_MORGUE;
    }
    else if (!zero_die(8) && (depth > 6))
    {
        zoo_style = ZOO_SHRINE;
    }
    else if (!zero_die(3))
    {
        zoo_style = ZOO_SMITHY;
    }
    else
    {
        zoo_style = NO_ZOO;
    }
    /* Add rooms */
    for (i = 0; i < actual_rooms; i++)
    {
	add_random_room(i / 3, i % 3);
    }
    /* Add corridors */
    /* Link the centre room to an edge room. */
    link_rooms(4, edge_rooms[zero_die(4)]);
    /* And to another; if we're already linked, don't bother. */
    i = zero_die(4);
    if (linkage[4][edge_rooms[i]] == 0)
    {
	link_rooms(4, edge_rooms[i]);
    }
    /* Link each edge room to one of its corner rooms. */
    for (i = 0; i < 4; i++)
    {
	link_rooms(edge_rooms[i], corners[i][zero_die(2)]);
    }
    /* At this point, 1-2 edge rooms and their attached corner rooms
     * have linkage to the centre. */
    /* Link each edge room to its unlinked corner if it is not 2-linked
     * to the centre. */
    for (i = 0; i < 4; i++)
    {
	if (!linkage[4][edge_rooms[i]])
	{
	    if (linkage[edge_rooms[i]][corners[i][0]])
	    {
		link_rooms(edge_rooms[i], corners[i][1]);
	    }
	    else
	    {
		link_rooms(edge_rooms[i], corners[i][0]);
	    }
	}

    }
    /* Link each corner room to its unlinked edge if that edge is not
     * 2-linked to the centre.  If we still haven't got centre
     * connectivity for the edge room, connect the edge to the centre. */
    for (i = 0; i < 4; i++)
    {
	if (!linkage[4][edge_rooms[i]])
	{
	    if (!linkage[edge_rooms[i]][corners[i][0]])
	    {
		link_rooms(edge_rooms[i], corners[i][0]);
	    }
	    if (!linkage[edge_rooms[i]][corners[i][1]])
	    {
		link_rooms(edge_rooms[i], corners[i][1]);
	    }
	}
	if (!linkage[4][edge_rooms[i]])
	{
	    link_rooms(edge_rooms[i], 4);
	}
    }
    /* Just for safety's sake: Now we know all edges are attached,
     * make sure all the corners are. (Previously, it was possible
     * for them not to be. I know, because I met such a level :) */
    for (i = 3; i > -1; i--)
    {
        if ((!linkage[4][corners[i][0]]) &&
            (linkage[edge_rooms[i]][corners[i][0]] != 1))
	{
	    link_rooms(edge_rooms[i], corners[i][0]);
	}
        if ((!linkage[4][corners[i][1]]) &&
            (linkage[edge_rooms[i]][corners[i][1]] != 1))
	{
	    link_rooms(edge_rooms[i], corners[i][1]);
	}
    }
    /* Add the stairs */
    put_stairs();
}

void Levext_rooms::populate_zoo()
{
    int mons;
    int items;
    int tries;
    Mon_handle mon;
    Obj_handle obj;
    libmrl::Coord pos;
    /* A treasure zoo should get nine monsters and nine items. */
    for (mons = 0; mons < 9; mons++)
    {
	for (tries = 0; tries < 200; tries++)
	{
	    pos.y = exclusive_flat(bounds[zoo_room][0].y, bounds[zoo_room][1].y);
	    pos.x = exclusive_flat(bounds[zoo_room][0].x, bounds[zoo_room][1].x);
	    if (parent->monster_at(pos) == NO_MONSTER)
	    {
		mon = create_mon(NO_PM, pos);
		if (mon.valid())
		{
		    break;
		}
	    }
	}
    }
    for (items = 0; items < 9; items++)
    {
	for (tries = 0; tries < 200; tries++)
	{
	    pos.y = exclusive_flat(bounds[zoo_room][0].y, bounds[zoo_room][1].y);
	    pos.x = exclusive_flat(bounds[zoo_room][0].x, bounds[zoo_room][1].x);
	    if (parent->object_at(pos) == NO_OBJECT)
	    {
		obj = create_obj(NO_POBJ, 1, 0, pos);
		if (obj.valid())
		{
		    break;
		}
	    }
	}
    }
}

libmrl::Coord Levext_rooms::get_room_cell(int room) const
{
    libmrl::Coord tmp;
    tmp.y = exclusive_flat(bounds[room][0].y, bounds[room][1].y);
    tmp.x = exclusive_flat(bounds[room][0].x, bounds[room][1].x);
    return tmp;
}

int Levext_rooms::get_levgen_mon_spot(libmrl::Coord *ppos) const
{
    /* Get a vacant floor cell that isn't in the treasure zoo. */
    int room_try;
    int cell_try;
    libmrl::Coord trypos = dbash::NOWHERE;
    int room;
    for (room_try = 0; room_try < (MAX_ROOMS * 2); room_try++)
    {
	room = zero_die(actual_rooms);
	if (room == zoo_room)
	{
	    continue;
	}
	for (cell_try = 0; cell_try < 200; cell_try++)
	{
            trypos = get_room_cell(room);
            Terrain_num t = parent->terrain_at(trypos);
	    if ((parent->monster_at(trypos).valid()) ||
                terrain_data[t].impassable ||
                terrain_data[t].feature)
	    {
                trypos = dbash::NOWHERE;
		continue;
	    }
	    break;
	}
	break;
    }
    if (trypos == dbash::NOWHERE)
    {
	return -1;
    }
    *ppos = trypos;
    return 0;
}

void Levext_rooms::populate(void)
{
    int i;
    int j;
    libmrl::Coord pos;
    int ic;
    /* Check for a "treasure zoo" */
    /* Generate some random monsters */
    for (i = 0; i < 10; i++)
    {
	j = get_levgen_mon_spot(&pos);
	if (j == -1)
	{
	    continue;
	}
	create_mon(NO_PM, pos);
    }
    ic = 3 + depth;
    if (ic > 40)
    {
	/* Never create more than 40 items. */
	ic = 40;
    }
    /* Generate some random treasure */
    for (i = 0; i < ic; i++)
    {
	j = get_levgen_mon_spot(&pos);
	if (j == -1)
	{
	    continue;
	}
	create_obj(NO_POBJ, 1, 0, pos);
    }
}

libmrl::Coord Levext_rooms::get_injection_point(Leventry_mode mode) const
{
    return ustairs_pos;
}

Levext_rooms::Levext_rooms() : Levextra(),
    actual_rooms(MAX_ROOMS),
    zoo_room(NO_ROOM),
    dstairs_room(NO_ROOM),
    ustairs_room(NO_ROOM)
{
    int i, j;
    parent = 0;
    for (i = 0; i < MAX_ROOMS; ++i)
    {
        for (j = 0; j < MAX_ROOMS; ++j)
        {
            linkage[i][j] = 0;
        }
        segsused[i] = false;
        bounds[i][0] = dbash::NOWHERE;
        bounds[i][1] = dbash::NOWHERE;
    }
}

void Levext_rooms::excavate_corridor_segment(libmrl::Coord c1, libmrl::Coord c2, bool door1, bool door2)
{
    libmrl::Coord sgn = libmrl::sign(c2 - c1);
    libmrl::Coord pos;
    parent->set_terrain(c1, door1 ? DOOR : FLOOR);
    parent->set_terrain(c2, door2 ? DOOR : FLOOR);
    for (pos = c1 + sgn; pos != c2; pos += sgn)
    {
        parent->set_terrain(pos, FLOOR);
    }
}

/* rooms.cc */
