/* loadsave.c
 * 
 * Copyright 2009 Martin Read
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "dunbash.hh"
#include "objects.hh"
#include "monsters.hh"
#include "combat.hh"
#include "loadsave.hh"
#include "vision.hh"
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <limits.h>
#include <string>
#include "cfgfile.hh"

bool save_wait;
bool reload_wait;
bool always_fsync;

static void rebuild_mapmons(void);
static void rebuild_mapobjs(void);

static void rebuild_mapmons(void)
{
    int i;
    for (i = 0; i < 100; i++)
    {
        if (monsters[i].used)
        {
            currlev->set_mon_at(monsters[i].pos, i);
        }
    }
}

static void rebuild_mapobjs(void)
{
    int i;
    for (i = 0; i < 100; i++)
    {
        if (objects[i].used && !objects[i].with_you)
        {
            currlev->set_obj_at(objects[i].pos, i);
        }
    }
}

static void deserialise_gamestate(FILE *fp)
{
    game_tick = deserialise_uint32(fp);
    dungeon = deserialise_uint32(fp);
    depth = deserialise_uint32(fp);
    deserialise(fp, rng_state, 5);
}

static void deserialise(FILE *fp, Player *ptmp)
{
    fread(ptmp->name, 1, 16, fp);
    ptmp->name[16] = '\0';
    ptmp->pos.y = deserialise_uint32(fp);
    ptmp->pos.x = deserialise_uint32(fp);
    ptmp->body = deserialise_uint32(fp);
    ptmp->bdam = deserialise_uint32(fp);
    ptmp->agility = deserialise_uint32(fp);
    ptmp->adam = deserialise_uint32(fp);
    ptmp->hpmax = deserialise_uint32(fp);
    ptmp->hpcur = deserialise_uint32(fp);
    ptmp->food = deserialise_uint32(fp);
    ptmp->experience = deserialise_uint32(fp);
    ptmp->protection = deserialise_uint32(fp);
    ptmp->leadfoot = deserialise_uint32(fp);
    ptmp->armourmelt = deserialise_uint32(fp);
    ptmp->speed = deserialise_uint32(fp);
    deserialise(fp, ptmp->resistances, DT_COUNT);
    ptmp->level = deserialise_uint32(fp);
    ptmp->gold = deserialise_uint32(fp);
    deserialise_ohandle_array(fp, ptmp->inventory, INVENTORY_SIZE);
    // The following are stored as inventory offsets
    ptmp->weapon = deserialise_uint32(fp);
    ptmp->armour = deserialise_uint32(fp);
    ptmp->ring = deserialise_uint32(fp);
}

void deserialise(FILE *fp, libmrl::Coord *c)
{
    c->y = int(deserialise_uint32(fp));
    c->x = int(deserialise_uint32(fp));
}

void deserialise_objects(FILE *fp)
{
    uint32_t count;
    uint32_t i;
    uint32_t oref;
    count = deserialise_uint32(fp);
    for (i = 0; i < count; ++i)
    {
        oref = deserialise_uint32(fp);
        objects[oref].used = true;
        deserialise(fp, objects + oref);
    }
}

void deserialise(FILE *fp, Obj *optr)
{
    optr->obj_id = int(deserialise_uint32(fp));
    optr->quan = int(deserialise_uint32(fp));
    optr->with_you = bool(deserialise_uint32(fp));
    deserialise(fp, &(optr->pos));
    optr->durability = int(deserialise_uint32(fp));
    optr->meta = int(deserialise_uint32(fp));
    optr->used = true;
}

void deserialise_ohandle_array(FILE *fp, Obj_handle *array, int count)
{
    int i;
    for (i = 0; i < count; ++i)
    {
        array[i] = Obj_handle(deserialise_uint32(fp));
    }
}

void deserialise_permobj_vars(FILE *fp)
{
    int i;
    for (i = 0; i < PO_COUNT; ++i)
    {
        permobjs[i].known = bool(deserialise_uint32(fp));
        permobjs[i].power = int(deserialise_uint32(fp));
    }
}

void deserialise_monsters(FILE *fp)
{
    uint32_t count = deserialise_uint32(fp);
    uint32_t i;
    uint32_t mref;
    for (i = 0; i < count; ++i)
    {
        mref = deserialise_uint32(fp);
        monsters[mref].used = true;
        deserialise(fp, monsters + mref);
    }
}

void deserialise(FILE *fp, Mon *mptr)
{
    libmrl::Coord c;
    int k;
    mptr->mon_id = deserialise_uint32(fp);
    deserialise(fp, &(mptr->pos));
    deserialise(fp, &(mptr->ai_lastpos));
    mptr->hpmax = deserialise_uint32(fp);
    mptr->hpcur = deserialise_uint32(fp);
    mptr->mtohit = deserialise_uint32(fp);
    mptr->rtohit = deserialise_uint32(fp);
    mptr->defence = deserialise_uint32(fp);
    mptr->mdam = deserialise_uint32(fp);
    mptr->rdam = deserialise_uint32(fp);
    mptr->awake = deserialise_uint32(fp);
    mptr->meta = deserialise_uint32(fp);
    k = deserialise_uint32(fp);
    if (k > 0)
    {
        mptr->name = (char *)malloc(k);
        fread(mptr->name, k, 1, fp);
    }
    else
    {
        mptr->name = 0;
    }
    deserialise(fp, &c);
    if (c != dbash::NOWHERE)
    {
        mptr->current_path = new Astar_path();
        do
        {
            mptr->current_path->push_back(c);
            deserialise(fp, &c);
        } while (c != dbash::NOWHERE);
    }
    else
    {
        mptr->current_path = 0;
    }
}

Level * deserialise_level(FILE *fp)
{
    int i;
    Level *lp;
    uint32_t ht;
    uint32_t wd;
    ht = deserialise_uint32(fp);
    wd = deserialise_uint32(fp);
    lp = new Level(ht, wd);
    lp->levtype = deserialise_uint32(fp);
    lp->build();
    for (i = 0; i < lp->height; ++i)
    {
        deserialise(fp, lp->mflags[i], lp->width);
    }
    // save terrain
    for (i = 0; i < lp->height; ++i)
    {
        deserialise(fp, (uint32_t *) lp->terrain[i], lp->width);
    }
    // save regionnums
    for (i = 0; i < lp->height; ++i)
    {
        deserialise(fp, (uint32_t *) lp->rnums[i], lp->width);
    }
    return lp;
}

static void serialise(FILE *fp, Player *ptmp)
{
    fwrite(ptmp->name, 1, 16, fp);
    serialise(fp, uint32_t(ptmp->pos.y));
    serialise(fp, uint32_t(ptmp->pos.x));
    serialise(fp, uint32_t(ptmp->body));
    serialise(fp, uint32_t(ptmp->bdam));
    serialise(fp, uint32_t(ptmp->agility));
    serialise(fp, uint32_t(ptmp->adam));
    serialise(fp, uint32_t(ptmp->hpmax));
    serialise(fp, uint32_t(ptmp->hpcur));
    serialise(fp, uint32_t(ptmp->food));
    serialise(fp, ptmp->experience);
    serialise(fp, uint32_t(ptmp->protection));
    serialise(fp, uint32_t(ptmp->leadfoot));
    serialise(fp, uint32_t(ptmp->armourmelt));
    serialise(fp, uint32_t(ptmp->speed));
    serialise(fp, ptmp->resistances, DT_COUNT);
    serialise(fp, uint32_t(ptmp->level));
    serialise(fp, uint32_t(ptmp->gold));
    serialise_ohandle_array(fp, ptmp->inventory, INVENTORY_SIZE);
    serialise(fp, uint32_t(ptmp->weapon.value));
    serialise(fp, uint32_t(ptmp->armour.value));
    serialise(fp, uint32_t(ptmp->ring.value));
}

void serialise(FILE *fp, Obj const *optr)
{
    serialise(fp, uint32_t(optr->obj_id));
    serialise(fp, uint32_t(optr->quan));
    serialise(fp, uint32_t(optr->with_you));
    serialise(fp, optr->pos);
    serialise(fp, uint32_t(optr->durability));
    serialise(fp, uint32_t(optr->meta));
}

void serialise_objects(FILE *fp)
{
    int i;
    int count;
    for (i = count = 0; i < MAX_OBJECTS; ++i)
    {
        if (objects[i].used)
        {
            ++count;
        }
    }
    serialise(fp, uint32_t(count));
    for (i = 0; i < MAX_OBJECTS; ++i)
    {
        if (objects[i].used)
        {
            serialise_ohandle(fp, i);
            serialise(fp, objects + i);
        }
    }
}

void serialise_ohandle_array(FILE *fp, Obj_handle const *array, int count)
{
    int i;
    for (i = 0; i < count; ++i)
    {
        serialise(fp, uint32_t(array[i].value));
    }
}

static void serialise_gamestate(FILE *fp)
{
    serialise(fp, game_tick);
    serialise(fp, uint32_t(dungeon));
    serialise(fp, uint32_t(depth));
    serialise(fp, rng_state, 5);
}

void serialise(FILE *fp, libmrl::Coord c)
{
    serialise(fp, uint32_t(c.y));
    serialise(fp, uint32_t(c.x));
}

void serialise(FILE *fp, Level const *lp)
{
    int i;
    serialise(fp, uint32_t(lp->height));
    serialise(fp, uint32_t(lp->width));
    serialise(fp, uint32_t(lp->levtype));
    // skip mobjs
    // skip mmons
    // skip astar parameters
    // save flags
    for (i = 0; i < lp->height; ++i)
    {
        serialise(fp, lp->mflags[i], lp->width);
    }
    // save terrain
    for (i = 0; i < lp->height; ++i)
    {
        serialise(fp, (uint32_t *) lp->terrain[i], lp->width);
    }
    // save regionnums
    for (i = 0; i < lp->height; ++i)
    {
        serialise(fp, (uint32_t *) lp->rnums[i], lp->width);
    }
}

void serialise(FILE *fp, Mon const *mptr)
{
    Astar_path::iterator iter;
    serialise(fp, uint32_t(mptr->mon_id));
    serialise(fp, mptr->pos);
    serialise(fp, mptr->ai_lastpos);
    serialise(fp, uint32_t(mptr->hpmax));
    serialise(fp, uint32_t(mptr->hpcur));
    serialise(fp, uint32_t(mptr->mtohit));
    serialise(fp, uint32_t(mptr->rtohit));
    serialise(fp, uint32_t(mptr->defence));
    serialise(fp, uint32_t(mptr->mdam));
    serialise(fp, uint32_t(mptr->rdam));
    serialise(fp, uint32_t(mptr->awake));
    serialise(fp, uint32_t(mptr->meta));
    if (mptr->name)
    {
        int k = strlen(mptr->name) + 1;
        serialise(fp, uint32_t(strlen(mptr->name)));
        fwrite(mptr->name, k, 1, fp);
    }
    else
    {
        serialise(fp, uint32_t(0));
    }
    if (mptr->current_path)
    {
        for (iter = mptr->current_path->begin();
             iter != mptr->current_path->end();
             ++iter)
        {
            serialise(fp, *iter);
        }
    }
    serialise(fp, dbash::NOWHERE);
}

void serialise_monsters(FILE *fp)
{
    int i;
    int count = 0;
    for (i = 0; i < MAX_MONSTERS; ++i)
    {
        if (monsters[i].used)
        {
            ++count;
        }
    }
    serialise(fp, uint32_t(count));
    for (i = 0; i < MAX_MONSTERS; ++i)
    {
        if (monsters[i].used)
        {
            serialise(fp, uint32_t(i));
            serialise(fp, monsters + i);
        }
    }
}

void serialise_permobj_vars(FILE *fp)
{
    int i;
    for (i = 0; i < PO_COUNT; ++i)
    {
        serialise(fp, uint32_t(permobjs[i].known));
        serialise(fp, uint32_t(permobjs[i].power));
    }
}

void save_game(void)
{
    FILE *fp;
    int fd;
    int i;
    std::string filename;
#ifdef MULTIUSER
    char uidbuf[16];
    filename = configured_system_playground;
    filename += "/save/dunbash-";
    sprintf(uidbuf, "%x", user_uid);
    filename += uidbuf;
    filename += ".sav";
    game_permissions();
#else
    filename = "dunbash.sav";
#endif
    fd = open(filename.c_str(), O_WRONLY | O_EXCL | O_CREAT, S_IRUSR | S_IWUSR);
    if (fd == -1)
    {
        print_msg(MSGCHAN_INTERROR, "could not save to %s: %s\n", filename.c_str(),
                  strerror(errno));
        press_enter();
        return;
    }
    fp = fdopen(fd, "wb");
    serialise_gamestate(fp);
    serialise(fp, &u);
    serialise(fp, currlev);
    serialise_monsters(fp);
    serialise_objects(fp);
    serialise_permobj_vars(fp);
    fflush(fp);
    if (always_fsync)
    {
        fsync(fileno(fp));
    }
    fclose(fp);
    /* Compress! */
    std::string command;
    command = COMPRESSOR;
    command += " ";
    command += filename;
    i = system(command.c_str());
    if (i == 0)
    {
        print_msg(0, "Game saved; exiting.\n");
        game_finished = true;
    }
    if (save_wait)
    {
        press_enter();
    }
    return;
}

int load_game(void)
{
    FILE *fp;
    struct stat st;
    std::string command;
    std::string filename;
    std::string compressed_filename;
    int i;
#ifdef MULTIUSER
    char uidbuf[16];
    filename = configured_system_playground;
    filename += "/save/dunbash-";
    sprintf(uidbuf, "%x", getuid());
    filename += uidbuf;
    game_permissions();
#else
    filename = "dunbash";
#endif
    filename += ".sav";
    compressed_filename = filename;
    compressed_filename += COMPRESSED_SUFFIX;
    i = stat(compressed_filename.c_str(), &st);
    if (i != -1)
    {
        command = DECOMPRESSOR;
        command += " ";
        command += compressed_filename;
        i = system(command.c_str());
        if (i != 0)
        {
            print_msg(MSGCHAN_INTERROR, "compressed save file found but unable to decompress. Giving up...\n");
            press_enter();
            exit(1);
        }
    }
    i = stat(filename.c_str(), &st);
    if (i != -1)
    {
        fp = fopen(filename.c_str(), "rb");
        deserialise_gamestate(fp);
        deserialise(fp, &u);
        currlev = deserialise_level(fp);
        deserialise_monsters(fp);
        deserialise_objects(fp);
        deserialise_permobj_vars(fp);
        fclose(fp);
        rebuild_mapmons();
        rebuild_mapobjs();
        unlink(filename.c_str());
        touch_back_buffer();
        do_vision();
        status_updated = 1;
        map_updated = 1;
        hard_redraw = 1;
        recalc_defence();
        print_msg(0, "Game loaded.\n");
        if (reload_wait)
        {
            press_enter();
        }
        look_at_floor();
        return 0;
    }
    return -1;
}

/* loadsave.c */
