/* dunbash.hh
 * 
 * Copyright 2009 Martin Read
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef DUNBASH_HH
#define DUNBASH_HH

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>

#include "dbashcfg.hh"

#ifndef COORD_HH
#include "coord.hh"
#endif

namespace dbash
{
    extern const libmrl::Coord NOWHERE;
    extern const libmrl::Coord NORTH;
    extern const libmrl::Coord WEST;
    extern const libmrl::Coord EAST;
    extern const libmrl::Coord SOUTH;
    extern const libmrl::Coord NORTHEAST;
    extern const libmrl::Coord NORTHWEST;
    extern const libmrl::Coord SOUTHEAST;
    extern const libmrl::Coord SOUTHWEST;
    extern const libmrl::Coord NOWHERE;
}

#ifndef INDIE_HH
#include "indie.hh"
#endif

#ifndef POBJ1_HH
#include "pobj1.hh"
#endif

#ifndef PMON1_HH
#include "pmon1.hh"
#endif

/* XXX main.cc data and funcs */
#ifdef MULTIUSER
extern int user_permissions(void);
extern int game_permissions(void);
#endif
extern int exclusive_flat(int lower, int upper); /* l+1 ... u-1 */
extern int inclusive_flat(int lower, int upper); /* l ... u */
extern int one_die(int sides);	/* 1..n */
extern int dice(int count, int sides);
extern int zero_die(int sides);	/* 0..n-1 */
extern int do_command(Game_cmd command);
extern unsigned int convert_range(int dy, int dx);
extern bool game_finished;
extern uint32_t game_tick;
extern bool wizard_mode;
extern int game_uid;
extern int user_uid;
extern int action_speed;

/* XXX map.c data and funcs*/
extern void make_new_level(void);
extern void inject_player(void);
extern int get_room_x(int room);
extern int get_room_y(int room);
extern void room_reset(void);

enum Astar_states
{
    ASTAR_UNCONSIDERED,
    ASTAR_OPEN,
    ASTAR_CLOSED,
    ASTAR_REJECTED,
};

#define MAPFLAG_EXPLORED 0x00000001
#define MAPFLAG_NOPIERCE 0x00000002

enum Level_type
{
    LEVEL_ROOMS,
    LEVEL_CAVE,
    LEVEL_NECROPOLIS,
    LEVEL_INFERNO,
    TOTAL_LEVEL_STYLES
};

// Standard dimensions of a level
#define DUN_WIDTH 42
#define DUN_HEIGHT 42
// Maximum dimensions of a level, relied on by display.cc
#define MAX_DUN_WIDTH 84
#define MAX_DUN_HEIGHT 84
#define ROOM_HT_DELTA 4
#define ROOM_WD_DELTA 4

struct Level;

enum Leventry_mode
{
    // allow for up to three prime staircases
    Leventry_stairs_dn_1,
    Leventry_stairs_dn_2,
    Leventry_stairs_dn_3,
    Leventry_stairs_up_1,
    Leventry_stairs_up_2,
    Leventry_stairs_up_3,
    // hell entry and exit modes are special
    Leventry_portal_to_hell,
    Leventry_portal_from_hell
};

struct Levextra
{
    Level *parent;
    bool overridden_monsel;

    virtual void excavate(void) = 0;
    virtual void populate(void) = 0;
    virtual libmrl::Coord get_injection_point(Leventry_mode mode = Leventry_stairs_dn_1) const = 0;
    bool has_monsel_override(void) const { return overridden_monsel;}
};

struct Region
{
    virtual bool contains(libmrl::Coord c) const = 0;
    virtual bool linked_to(int rnum) const = 0;
    virtual libmrl::Coord random_point() const = 0;
    virtual ~Region() { }
};

#define NO_REGION -1
#include "monsters.hh"
#include "objects.hh"
struct Level
{
    int height;
    int width;
    int levtype;
    Obj_handle **mobjs;
    Mon_handle **mmons;
    Terrain_num **terrain;
    int32_t **rnums; /* region numbers */
    uint32_t **astar_invoc_num;
    uint32_t **astar_considered;
    libmrl::Coord **came_from;
    uint32_t **gscores;
    uint32_t **mflags;
    Levextra *levextra;
    Level() { }
    Level(int ht, int wd);
    ~Level();
    void build(void);
    void excavate(void);
    void populate(void);
    void leave(void);
    void release(void);
    bool outofbounds(libmrl::Coord c) const { return (c.y < 0) || (c.x < 0) || (c.y >= height) || (c.x >= width); }
    Terrain_num terrain_at(libmrl::Coord c) const { return terrain[c.y][c.x]; }
    void set_terrain(libmrl::Coord c, Terrain_num t) const { terrain[c.y][c.x] = t; }
    uint32_t flags_at(libmrl::Coord c) const { return mflags[c.y][c.x]; }
    void set_flag_at(libmrl::Coord c, uint32_t f) const { mflags[c.y][c.x] |= f; }
    void clear_flag_at(libmrl::Coord c, uint32_t f) const { mflags[c.y][c.x] &= ~f; }
    Mon_handle monster_at(libmrl::Coord c) const { return mmons[c.y][c.x]; }
    Obj_handle object_at(libmrl::Coord c) const { return mobjs[c.y][c.x]; }
    void set_mon_at(libmrl::Coord c, Mon_handle m) { mmons[c.y][c.x] = m; }
    void set_obj_at(libmrl::Coord c, Obj_handle o) { mobjs[c.y][c.x] = o; }
    int region_at(libmrl::Coord c) const { return rnums[c.y][c.x]; }
    void set_region(libmrl::Coord c, int r) const { rnums[c.y][c.x] = r; }
    int as_gscore(libmrl::Coord c) const { return gscores[c.y][c.x]; }
    int as_invoc(libmrl::Coord c) const { return astar_invoc_num[c.y][c.x]; }
    libmrl::Coord as_came_from(libmrl::Coord c) const { return came_from[c.y][c.x]; }
    int as_considered(libmrl::Coord c) const { return astar_considered[c.y][c.x]; }
    void as_considered(libmrl::Coord c, int val) { astar_considered[c.y][c.x] = val; }
    void set_as_gscore(libmrl::Coord c, int val) { gscores[c.y][c.x] = val; }
    void set_as_invoc(libmrl::Coord c, int val) { astar_invoc_num[c.y][c.x] = val; }
    void set_as_came_from(libmrl::Coord c, libmrl::Coord val) { came_from[c.y][c.x] = val; }
    void set_as_considered(libmrl::Coord c, Astar_states val) { astar_considered[c.y][c.x] = val; }
    Terrain_num base_floor(void) const { return FLOOR; }
    Terrain_num base_wall(void) const { return WALL; }
};

extern Level *currlev;
extern uint32_t depth;
extern uint32_t dungeon;

typedef int (*Mon_select_fun)(int levnum);

struct Dungeon_desc
{
    char const *name;
    bool persistent_levels;
    Mon_select_fun mon_selector;
};

extern Dungeon_desc dungeon_specs[];

struct Terrain_desc
{
    char const *name;
    bool opaque;        // blocks LOS
    bool impassable;    // blocks movement and projectiles
    bool feature;       // is a dungeon feature
    bool hostile;       // is hostile
    char symbol;        // symbol for text-mode display
    Dbash_colour colour;        // colour for text-mode display
};

extern const Terrain_desc terrain_data[];

/* XXX misc.c data and funcs */
extern const char *damtype_names[DT_COUNT];
extern void get_iso_8601_time(std::string& dest);

/* XXX objects.c data and funcs */
extern void flavours_init(void);

/* XXX rng.c data and funcs */
#define RNG_MAX 0xFFFFFFFFu
extern uint32_t rng_state[5];
extern uint32_t saved_state[5];
extern uint32_t rng(void);
extern void rng_init(void);

/* XXX vector.c data and funcs */
extern void compute_directions(libmrl::Coord c1, libmrl::Coord c2, Direction_data *dir_data); //int y1, int x1, int y2, int x2, int *pdy, int *pdx, int *psy, int *psx, int *pmelee, int *pcardinal);

extern bool pos_visible(libmrl::Coord pos);

extern void ptac_init(void);
extern char *demon_get_name(void);
extern const char *numberwords[40];
#ifndef DISPLAY_H
#include "display.hh"
#endif

#ifndef PLAYER_H
#include "player.hh"
#endif

#endif

/* dunbash.hh */
